/* 
 * routines for computing analysis heatmaps 
 *
 * Copyright 1994, 1995 Bill Press and Bruno Olshausen
 * Washington University School of Medicine
 *
 */

#include <stdio.h>
#include <malloc.h>
#include <math.h>

#include "xanat.h"

#define PROB_WEIGHT	1.0

/*
 * timing code
 
 #include <sys/time.h>
 struct timeval tp1,tp2;
 struct timezone tzp;
 
 double time_diff(struct timeval	*tp1, struct timeval *tp2)
 {
   int	start,finish;

   start=(tp1->tv_sec%100)*1000000 + tp1->tv_usec;
   finish=(tp2->tv_sec%100)*1000000 + tp2->tv_usec;

   return((double)(finish-start)/1000000);
 }
 */


/**********  Shared routines  **********/

int findOneArea (Object *obj, int mode, int inj)
{
  int 	totArea = 0;
    
  while (obj) {
    if (mode==SEARCH_AREA)
      totArea += obj->header.size;
    else if (obj->header.type==mode && obj->header.inj_num==inj &&
	     inject_toggles[obj->header.inj_num])
      totArea += obj->header.size * (obj->header.strength+1);
    obj=obj->header.next;
  }
  return(totArea);
}



int findTotArea (Object **objectList, int mode, int inj)
{
  int 	totArea;
  int 	plane;
    
  for ( plane=0, totArea=0; plane<num_images; plane++ )
    totArea += findOneArea (objectList[plane], mode, inj);

  return(totArea);
}



void initHeatMap (double **heatMap)
{
  int		plane, i;
  double	*heatMapPointer;

  for ( plane=0; plane<num_images; plane++ ) {

    heatMapPointer = heatMap[plane];

    for ( i=0; i<icon[plane].nrc; i++ )
      *heatMapPointer++ = 0.0;
  }
}


void initListPointers (Datum *data)
{
  Datum *currentDatum;
    
  for (currentDatum = data; currentDatum != NULL; 
       currentDatum=currentDatum->next[DATA])
    currentDatum->inAnalysis = 0;
}



void resetSearchList(Datum *data)
{
  Datum   *currentDatum;
  Datum   *firstDatum;
  Datum   *lastDatum;

  int	    foundFirst = 0;
    
    
  for ( currentDatum = data; currentDatum!=NULL;
       currentDatum=currentDatum->next[DATA]) {
	
    if (currentDatum->inAnalysis)
			
      if (!foundFirst) {
	currentDatum->prev[SEARCH] = NULL;
	currentDatum->next[SEARCH] = NULL;
	firstDatum = currentDatum;
	lastDatum = currentDatum;
	foundFirst = 1;
      } else {
	currentDatum->prev[SEARCH] = lastDatum;
	currentDatum->next[SEARCH] = NULL;
	lastDatum->next[SEARCH] = currentDatum;
	lastDatum = currentDatum;
      }
  }
  if (foundFirst)
    list[SEARCH] = firstDatum;
  else
    list[SEARCH] = NULL;
}


/**********  Superposition routines  **********/


void doSuperposition (Datum *currentDatum, double **heatMap, int drawType, 
		      int inj, int searchPlane, Box *searchBox, int normArea)
{
  double	*heatMapPointer, *hptr;
    
  double	overlap;

  int		i, j, offset;
  int		resultPlane;

  Object	**datumObjects;
  Box		resultBox;
  int		resultPixelWeight;

  datumObjects=currentDatum->objects;

  overlap = 0.0;

  if (objects_within_bounding_box_p(datumObjects[searchPlane], searchBox)) {

    draw_object_list_in_icon_pixmap(datumObjects[searchPlane],
				    drawType, inj, 1, 1);
    get_data_ximage();
    overlap=(double)region_weighted_overlap(searchBox);
  }

  if (overlap!=0.0) {

    currentDatum->inAnalysis = 1;

    overlap *= overlap/(double)normArea;

    if (currentDatum->confidence != 0)
      overlap *= (double)currentDatum->confidence;
    else
      overlap *= 100.0;

    for (resultPlane=0; resultPlane<num_images; resultPlane++) {

      switch (drawType) {
			
       case LABEL:
	draw_object_list_in_icon_pixmap (datumObjects[resultPlane],
					 INJECTION, inj, 1, 1);
	compute_object_list_icon_bounding_box(datumObjects[resultPlane], 
					      INJECTION, &resultBox);
	break;
       case INJECTION:
	draw_object_list_in_icon_pixmap(datumObjects[resultPlane],
					LABEL, inj, 1, 1);
	compute_object_list_icon_bounding_box(datumObjects[resultPlane],
					      LABEL, &resultBox);
	break;
      }

      get_data_ximage();
			
      offset=resultBox.ul.y*icon[resultPlane].width + resultBox.ul.x;
      heatMapPointer = heatMap[resultPlane]+offset;

      for ( j=resultBox.ul.y; j<resultBox.lr.y; j++ ) {

	hptr=heatMapPointer;

	for ( i=resultBox.ul.x; i<resultBox.lr.x; i++, hptr++ )
	  if (resultPixelWeight=data_ximage_pixel(i,j))
	    *hptr += overlap*resultPixelWeight;

	heatMapPointer += icon[resultPlane].width;
      }
    }
  }
}



int analyzeSuperpos(Datum *analysis_datum, Datum *data, int direction)
{
  Datum	*currentDatum;
  Box	searchBox;
  int	plane;
  int	labelSize, injectSize;
  int	inj;

  Object **objectList=analysis_datum->objects;
  double **heatMap=analysis_datum->heatmap;    

  if (findTotArea (objectList, SEARCH_AREA, NUM_DIFF_INJ) == 0) {
	
    perr("Search area too small.");
    return -1;
  }

  initHeatMap (heatMap);
  initListPointers (data);

  pmes("Analyzing...");

  for ( plane=0; plane<num_images; plane++ )

    if (objectList[plane] != NULL) {
    
      draw_object_list_in_icon_pixmap(objectList[plane],
				      SEARCH_AREA, NUM_DIFF_INJ, 1, 1);
      get_search_ximage();
      compute_object_list_icon_bounding_box(objectList[plane],
					    SEARCH_AREA, &searchBox);

      for (currentDatum=data; currentDatum!=NULL;
	   currentDatum=currentDatum->next[DATA])

	for (inj=0; inj<NUM_DIFF_INJ; inj++) {

	  labelSize = findTotArea (currentDatum->objects, LABEL, inj);
	  injectSize = findTotArea (currentDatum->objects, INJECTION, inj);

	  if ((direction==TO || direction==FROM) && currentDatum->labelType) {
	    if ( (direction==TO && currentDatum->labelType==ANTEROGRADE) ||
		(direction==FROM && currentDatum->labelType==RETROGRADE))
	      doSuperposition (currentDatum, heatMap, LABEL, inj,
			       plane, &searchBox, labelSize);
	    else
	      doSuperposition (currentDatum, heatMap, INJECTION, inj,
			       plane, &searchBox, injectSize);
	  } else {
	    doSuperposition (currentDatum, heatMap, LABEL, inj,
			     plane, &searchBox, labelSize);
	    doSuperposition (currentDatum, heatMap, INJECTION, inj,
			     plane, &searchBox, injectSize);
	  }
	}
    }

  analysis_datum->analysis_midpoint = 0;
  find_minmax(&(analysis_datum->heatmap_min),
	      &(analysis_datum->heatmap_max),
	      analysis_datum->heatmap);
  rescaleHeatMap(analysis_datum->heatmap,
		 analysis_datum->heatmap_min,
		 analysis_datum->heatmap_max);
  find_minmax(&(analysis_datum->heatmap_min),
	      &(analysis_datum->heatmap_max),
	      analysis_datum->heatmap);
   
  resetSearchList(data);

  return 0;
}



int analyzeByListSuperpos(Datum *analysis_datum, Datum *data, int direction)
{
  Datum	*currentDatum;    
  Box	searchBox;
  int	plane;
  int	labelSize, injectSize;
  int	inj;
    
  Object **objectList=analysis_datum->objects;
  double **heatMap=analysis_datum->heatmap;    

  if (findTotArea (objectList, SEARCH_AREA, NUM_DIFF_INJ) == 0) {
	
    perr("Search area too small.");
    return -1;
  }

  initHeatMap (heatMap);

  pmes("Analyzing...");

  for ( plane=0; plane<num_images; plane++ )

    if (objectList[plane] != NULL) {
			
      draw_object_list_in_icon_pixmap(objectList[plane],
				      SEARCH_AREA, NUM_DIFF_INJ, 1, 1);
      get_search_ximage();
      compute_object_list_icon_bounding_box(objectList[plane],
					    SEARCH_AREA, &searchBox);

      for (currentDatum=data; currentDatum!=NULL;
	   currentDatum=currentDatum->next[SEARCH])

	for (inj=0; inj<NUM_DIFF_INJ; inj++) {

	  labelSize = findTotArea (currentDatum->objects, LABEL, inj);
	  injectSize = findTotArea (currentDatum->objects, INJECTION, inj);

	  if ((direction==TO || direction==FROM) && currentDatum->labelType) {
	    if ( (direction==TO && currentDatum->labelType==ANTEROGRADE) ||
		(direction==FROM && currentDatum->labelType==RETROGRADE))
	      doSuperposition(currentDatum, heatMap, LABEL, inj,
			      plane, &searchBox, labelSize);
	    else
	      doSuperposition(currentDatum, heatMap, INJECTION, inj,
			      plane, &searchBox, injectSize);
	  } else {
	    doSuperposition(currentDatum, heatMap, LABEL, inj,
			    plane, &searchBox, labelSize);
	    doSuperposition(currentDatum, heatMap, INJECTION, inj,
			    plane, &searchBox, injectSize);
	  }
	}
    }

  analysis_datum->analysis_midpoint = 0;
  find_minmax(&(analysis_datum->heatmap_min),
	      &(analysis_datum->heatmap_max),
	      analysis_datum->heatmap);
  rescaleHeatMap(analysis_datum->heatmap,
		 analysis_datum->heatmap_min,
		 analysis_datum->heatmap_max);
  find_minmax(&(analysis_datum->heatmap_min),
	      &(analysis_datum->heatmap_max),
	      analysis_datum->heatmap);

  return 0;
}




/**********  Bayes routines  **********/



void initAnalyzeMatrices (double **analyzeMatrix0, double **analyzeMatrix1)
{
  double *analyzePointer0;
  double *analyzePointer1;
    
  int plane, i;

  for ( plane=0; plane<num_images; plane++ ) {

    analyzePointer0 = analyzeMatrix0[plane];
    analyzePointer1 = analyzeMatrix1[plane];

    for ( i=0; i<icon[plane].nrc; i++ ) {
	
      *analyzePointer0++ = 0.5;
      *analyzePointer1++ = 0.5;
    }
  }
}




int inObject (Object *objects, int x, int y, int drawType)
{
  int count = 0;

  x*=ICON_SF;
  y*=ICON_SF;
    
  while (objects) {
	
    if ( within_object_p (objects, x, y) &&
	objects->header.type==drawType &&
	inject_toggles[objects->header.inj_num])
      count++;

    objects = objects->header.next;
  }
    
  if (!count)
    return (0);
  else
    return (1);    
}


void addToHeatMap(double **heatMap, double **analyzeMatrix0,
		  double **analyzeMatrix1)
{
  int plane, i;
    
  double *heatMapPointer;
  double *analyzePointer0;
  double *analyzePointer1;
    
  for ( plane=0; plane<num_images; plane++ ) {
	    
    analyzePointer0 = analyzeMatrix0[plane];
    analyzePointer1 = analyzeMatrix1[plane];
    heatMapPointer  = heatMap[plane];
		
    for ( i=0; i<icon[plane].nrc; i++ ) {
		
      *analyzePointer1 /= *analyzePointer0++ + *analyzePointer1;
      *heatMapPointer++ += *analyzePointer1++;
    }
  }
}


void rescaleBayesHeatMap (double **heatMap, int numSearch)
{
  int		plane, i;
  double	*heatMapPointer;
    
  for ( plane=0; plane<num_images; plane++ ) {

    heatMapPointer = heatMap[plane];
	
    for ( i=0; i<icon[plane].nrc; i++ )
      *heatMapPointer++ /= numSearch;
  }
}



void doBayes (Datum *currentDatum, double **analyzeMatrix0,
	      double **analyzeMatrix1, int drawType,
	      int searchPlane, int x, int y)
{
  double	confidence;
  int		isInObject;

  int		resultPlane;
  int		i, j;

  int		inj;
    
  double	*analyzePointer0;
  double	*analyzePointer1;
    
  confidence = ((double) currentDatum->confidence / 2 + 50.0) / 100.0;

  isInObject = inObject (currentDatum->objects[searchPlane], x, y, drawType);

  if (drawType==LABEL || (drawType==INJECTION && isInObject)) {

    if (drawType==INJECTION)
      currentDatum->inAnalysis = 1;
    
    for ( resultPlane=0; resultPlane<num_images; resultPlane++ )

      if (currentDatum->validSlice[resultPlane]) {

	switch (drawType) {
			
	 case LABEL:
	  for (inj=0, clear_icon_pixmap(); inj<NUM_DIFF_INJ; inj++)
	    draw_object_list_in_icon_pixmap
	      (currentDatum->objects[resultPlane], INJECTION, inj, 1, 0);
	  break;
	 case INJECTION:
	  for (inj=0, clear_icon_pixmap(); inj<NUM_DIFF_INJ; inj++)
	    draw_object_list_in_icon_pixmap
	      (currentDatum->objects[resultPlane], LABEL, inj, 1, 0);
	  break;
	}

	get_data_ximage();
    
	analyzePointer0 = analyzeMatrix0[resultPlane];
	analyzePointer1 = analyzeMatrix1[resultPlane];

	for (j=0; j<icon[resultPlane].height; j++)
	  for (i=0; i<icon[resultPlane].width; i++) {
			    
	    if (drawType == INJECTION) {
			
	      if (data_ximage_pixel(i,j)) {

		*analyzePointer0 *= 1.0 - confidence;
		*analyzePointer1 *= confidence;

	      } else {

		*analyzePointer0 *= confidence * PROB_WEIGHT;
		*analyzePointer1 *= (1.0 - confidence) * PROB_WEIGHT;
	      }
	    } else {
			
	      if (data_ximage_pixel(i,j)) {			
			
		if (isInObject) {

		  *analyzePointer0 *= 1.0 - confidence;
		  *analyzePointer1 *= confidence;
		  currentDatum->inAnalysis = 1;

		} else {

		  *analyzePointer0 *= confidence * PROB_WEIGHT;
		  *analyzePointer1 *= (1.0 - confidence) * PROB_WEIGHT;
		}
	      }
	    }

	    analyzePointer0++;
	    analyzePointer1++;
	  }
      }
  }
}


int analyzeBayes (Datum *analysis_datum, Datum *data, int direction)
{
  Datum *currentDatum;
    
  Object **objectList=analysis_datum->objects;
  double **heatMap=analysis_datum->heatmap;

  double **analyzeMatrix0;
  double **analyzeMatrix1;
    
  int plane, x,y;
  int numSearch, totSearch;
    
  char	*ch;
  char	leftString[80];

  extern char *query();

  if ((totSearch = findTotArea (objectList, SEARCH_AREA, NUM_DIFF_INJ)) == 0) {
	
    perr("Search area too small.");
    return -1;
  }

  if (totSearch > 10) {
    ch=query("Search area large for Bayes: continue? ");
    if (*ch=='n' || *ch=='N') {
      perr("Bayesian analysis aborted");
      return -1;
    }
  }

  analyzeMatrix0 = (double **) calloc (num_images, sizeof (double *));
  MCHECK(analyzeMatrix0);
  for ( plane=0; plane<num_images; plane++ ) {
    analyzeMatrix0[plane] = (double *) malloc(icon[plane].nrc*sizeof (double));
    MCHECK(analyzeMatrix0[plane]);
  }

  analyzeMatrix1 = (double **) calloc (num_images, sizeof (double *));
  MCHECK(analyzeMatrix1);
  for ( plane=0; plane<num_images; plane++ ) {
    analyzeMatrix1[plane] = (double *) malloc(icon[plane].nrc*sizeof (double));
    MCHECK(analyzeMatrix1[plane]);
  }

  initListPointers (data);
  initHeatMap (heatMap);
 
  pmes("Analysis 0% complete");
    
  for ( plane=0, numSearch=0; plane<num_images; plane++ )
    
    if (objectList[plane] != NULL) {
    
      draw_object_list_in_icon_pixmap(objectList[plane],
				      SEARCH_AREA, NUM_DIFF_INJ, 1, 1);
      get_search_ximage();
	
      for ( y=0; y<icon[plane].height; y++ ) {
	for ( x=0; x<icon[plane].width; x++ ) {

	  if (search_ximage_pixel(x,y)) {
	    
	    numSearch++;

	    initAnalyzeMatrices (analyzeMatrix0, analyzeMatrix1);

	    for (currentDatum=data; currentDatum!=NULL;
		 currentDatum=currentDatum->next[DATA]) {

	      if ((direction==TO || direction==FROM) &&
		  currentDatum->labelType) {
		if ( (direction==TO && currentDatum->labelType==ANTEROGRADE) ||
		    (direction==FROM && currentDatum->labelType==RETROGRADE))
		  doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			  LABEL, plane, x, y);
		else
		  doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			  INJECTION, plane, x, y);
	      } else {
		doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			LABEL, plane, x, y);
		doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			INJECTION, plane, x, y);
	      }
	    }

	    addToHeatMap(heatMap, analyzeMatrix0, analyzeMatrix1);

	    sprintf(leftString, "Analysis %.1lf%% complete",
		    100 * (double) numSearch / (double) totSearch);
	    pmes(leftString);
	  }
	}
      }
    }

  rescaleBayesHeatMap (heatMap, numSearch);
    
  analysis_datum->heatmap_min=0.0;
  analysis_datum->heatmap_max=1.0;

  analysis_datum->analysis_midpoint = 0.5;

  resetSearchList (data);

  return 0;
}


int analyzeByListBayes (Datum *analysis_datum, Datum *data, int direction)
{
  Datum *currentDatum;
    
  Object **objectList=analysis_datum->objects;
  double **heatMap=analysis_datum->heatmap;

  double **analyzeMatrix0;
  double **analyzeMatrix1;
    
  int plane, x,y;
  int numSearch, totSearch;
    
  char	*ch;
  char	leftString[80];

  extern char *query();

  if ((totSearch = findTotArea (objectList, SEARCH_AREA, NUM_DIFF_INJ)) == 0) {
	
    perr("Search area too small.");
    return -1;
  }

  if (totSearch > 10) {
    ch=query("Search area large for Bayes: continue? ");
    if (*ch=='n' || *ch=='N') {
      perr("Bayesian analysis aborted");
      return -1;
    }
  }
    
  analyzeMatrix0 = (double **) calloc (num_images, sizeof (double *));
  MCHECK(analyzeMatrix0);
  for ( plane=0; plane<num_images; plane++ ) {
    analyzeMatrix0[plane] = (double *) malloc(icon[plane].nrc*sizeof (double));
    MCHECK(analyzeMatrix0[plane]);
  }

  analyzeMatrix1 = (double **) calloc (num_images, sizeof (double *));
  MCHECK(analyzeMatrix1);
  for ( plane=0; plane<num_images; plane++ ) {
    analyzeMatrix1[plane] = (double *) malloc(icon[plane].nrc*sizeof (double));
    MCHECK(analyzeMatrix1[plane]);
  }

  initHeatMap (heatMap);
 
  pmes("Analysis 0% complete");
    
  for ( plane=0, numSearch=0; plane<num_images; plane++ )
    
    if (objectList[plane] != NULL) {
    
      draw_object_list_in_icon_pixmap(objectList[plane],
				      SEARCH_AREA, NUM_DIFF_INJ, 1, 1);
      get_search_ximage();
	
      for ( y=0; y<icon[plane].height; y++ ) {
	for ( x=0; x<icon[plane].width; x++ ) {

	  if (search_ximage_pixel(x,y)) {
	    
	    numSearch++;

	    initAnalyzeMatrices (analyzeMatrix0, analyzeMatrix1);

	    for (currentDatum=data; currentDatum!=NULL;
		 currentDatum=currentDatum->next[SEARCH]) {

	      if ((direction==TO || direction==FROM) &&
		  currentDatum->labelType) {
		if ( (direction==TO && currentDatum->labelType==ANTEROGRADE) ||
		    (direction==FROM && currentDatum->labelType==RETROGRADE))
		  doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			  LABEL, plane, x, y);
		else
		  doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			  INJECTION, plane, x, y);
	      } else {
		doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			LABEL, plane, x, y);
		doBayes(currentDatum, analyzeMatrix0, analyzeMatrix1,
			INJECTION, plane, x, y);
	      }
	    }

	    addToHeatMap(heatMap, analyzeMatrix0, analyzeMatrix1);

	    sprintf(leftString, "Analysis %.1lf%% complete",
		    100 * (double) numSearch / (double) totSearch);
	    pmes(leftString);
	  }
	}
      }
    }

  rescaleBayesHeatMap (heatMap, numSearch);

  analysis_datum->heatmap_min=0.0;
  analysis_datum->heatmap_max=1.0;

  analysis_datum->analysis_midpoint = 0.5;

  return 0;
}
