/*
 * reading (loading) and writing (saving) of datums
 *
 * Copyright 1994, 1995 Bill Press and Bruno Olshausen
 * Washington University School of Medicine
 *
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <malloc.h>

#include "xanat.h"
#include "io.h"

int writeData(Datum *data, char *name, int conversion)
{
  int	    	fd;
  Datum   	*currentDatum;
  Object  	*currentObject;
  char    	*ch,line[80];
  extern char	*query();

  int	    i, j, num_objects;

  double	version_num = DATA_FILE_VERSION;
  char	has_version = 1;
    
  if (conversion) {
    if ((fd = open(name, O_WRONLY|O_TRUNC, 0644))==-1)
      return(-1);
  }
  else
    if ((fd = open(name, O_WRONLY|O_CREAT|O_EXCL, 0644))==-1) {
      sprintf(line,"OK to overwrite file: %s ? ",name);
      ch=query(line);
      if (*ch=='y' || *ch=='Y') {
	if ((fd = open(name, O_WRONLY|O_TRUNC))==-1)
	  return(-1);
      } else
	return(-1);
    }

  write(fd, &has_version, 1);
  if (bigEndian)
    flip_double (&version_num);
  write(fd, &version_num, sizeof(double));

  for ( currentDatum=data; currentDatum!=NULL;
       currentDatum=currentDatum->next[DATA]) {

    if (bigEndian)
      flip_datum (currentDatum, 0);
	  
    write(fd, currentDatum->ref.full, CHARS_PER_REF);
    write(fd, currentDatum->comments, CHARS_PER_COMMENTS);
    write(fd, &(currentDatum->confidence), sizeof(int));
    write(fd, &(currentDatum->writeProtect), sizeof(int));

    write_areas(fd,currentDatum->injAreas);
    write_areas(fd,currentDatum->labelAreas);

    write(fd, &(currentDatum->labelType), sizeof(int));

    for ( i=0; i<num_images; i++ ) {

      num_objects=0;
      currentObject=currentDatum->objects[i];
      while(currentObject) {
	currentObject=currentObject->header.next;
	num_objects++;
      }
      if (bigEndian)
	flip_int(&num_objects);

      write(fd, &num_objects, sizeof(int));
      write(fd, &(currentDatum->validSlice[i]), sizeof(int));
      for (currentObject=currentDatum->objects[i];
	   currentObject!=NULL;
	   currentObject = currentObject->header.next )
	write(fd, currentObject, sizeof(Object));	      
    }
    if (bigEndian)
      flip_datum (currentDatum, 1);
  }
  
  close(fd);
  return(1);
}

write_areas(fd,datum_areas)
    int	fd;
    Area	*datum_areas;
{
  int	n;
  Area	*currentArea;

  n=0;
  currentArea=datum_areas;
  while(currentArea) {
    currentArea=currentArea->next;
    n++;
  }
  if (bigEndian)
    flip_int(&n);
  write(fd, &n, sizeof(int));
  
  for ( currentArea=datum_areas; currentArea!=NULL;
       currentArea=currentArea->next) {
    
    write(fd, currentArea->name, CHARS_PER_AREA_LABEL);
    write(fd, &(currentArea->fract), sizeof(double));
  }
}

Datum *readData(char *name)
{
  int		fd;
  Datum		*data = NULL;
  Datum		*currentDatum;
  Object	*currentObject;

  Object_pre2	*object_pre2;

  double	version_num;
  char		has_version;
  char		rest_ref[CHARS_PER_REF-1];

  int		i, j, k, num_objects;
  int		dummy_int;
  int		result;
  char		ref[CHARS_PER_REF];
  extern Area	*read_areas();
    
  if ((fd = open(name, O_RDONLY))==-1) {
    fprintf(stderr, "Could not open file.\n");
    return(NULL);
  }
    
  result = read(fd, ref, 1);

  if (result!=0) {
    has_version = ref[0];
    if (has_version==1) {
      read(fd, &version_num, sizeof(double));
      if (bigEndian)
	flip_double(&version_num);
      result = read(fd, ref, CHARS_PER_REF);
    } else {
      version_num = 1.0;
      result = read(fd, rest_ref, CHARS_PER_REF-1);
      strcat(ref, rest_ref);
    }
  }

  if (result!=0) {
    data = currentDatum = (Datum *) calloc(1, sizeof(Datum));
    MCHECK(data);
    init_datum(currentDatum, DATA_TYPE);
  }
    
  if (version_num!=DATA_FILE_VERSION)
    pmes("Converting...");

  while (result!=0) {
    
    strcpy(currentDatum->ref.full, ref);
    ref2abbrev(currentDatum,ref);
    read(fd, currentDatum->comments, CHARS_PER_COMMENTS);
    read(fd, &(currentDatum->confidence), sizeof(int));
    if (version_num>=1.2)
      read(fd, &(currentDatum->writeProtect), sizeof(int));

    currentDatum->injAreas=read_areas(fd);
    currentDatum->labelAreas=read_areas(fd);

    read(fd, &(currentDatum->labelType), sizeof(int));

    for ( i=0; i<num_images; i++ ) {
	
      if (version_num>=1.1 && version_num<2.0)
	for (k=0; k<4; k++)
	  read(fd, &(dummy_int), sizeof(int));

      read(fd, &num_objects, sizeof(int));
      if (bigEndian)
	flip_int (&num_objects);
      read(fd, &(currentDatum->validSlice[i]), sizeof(int));

      if (num_objects!=0) {
	currentDatum->objects[i] = (Object *)calloc(1, sizeof(Object));
	MCHECK(currentDatum->objects[i]);
      }
	
      for ( j=0, currentObject=currentDatum->objects[i]; j<num_objects; j++ ) {

	if (version_num<2.0) {
	  object_pre2=(Object_pre2 *)calloc(1,sizeof(Object_pre2));
	  MCHECK(object_pre2);
	  read(fd, object_pre2, sizeof(Object_pre2));
	  convert_object_pre2(currentObject, object_pre2, version_num, 1);
	  free(object_pre2);
	}
	else
	  read(fd, currentObject, sizeof(Object));
	    
	if (j+1<num_objects) {
	  currentObject->header.next = (Object *) calloc(1, sizeof(Object));
	  MCHECK(currentObject->header.next);
	  currentObject = currentObject->header.next;
	} else {
	  currentObject->header.next = NULL;
	}
      }
    }

    if (bigEndian)
      flip_datum (currentDatum, 1);

    if (version_num<1.2)
      currentDatum->writeProtect=1;

    /* compute_areas(currentDatum); */

    result = read(fd, ref, CHARS_PER_REF);
	
    if (result!=0) {
      currentDatum->next[DATA] = (Datum *) calloc(1, sizeof(Datum));
      MCHECK(currentDatum->next[DATA]);
      init_datum(currentDatum->next[DATA], DATA_TYPE);
      currentDatum->next[DATA]->prev[DATA] = currentDatum;
      currentDatum = currentDatum->next[DATA];
    } else {
      currentDatum->next[DATA] = NULL;
    }
  }
  close(fd);

  if (version_num!=DATA_FILE_VERSION)
    writeData(data,name,1);

  return(data);
}

Area *read_areas(fd)
    int	fd;
{
  int	i, j, n;
  Area	*return_list, *currentArea;

  read(fd, &n, sizeof(int));
  if (bigEndian)
    flip_int(&n);
  if (n!=0) {
    return_list = (Area *) calloc(1, sizeof(Area));
    MCHECK(return_list);
  }
  else
    return(NULL);

  for ( i=0, currentArea=return_list; i<n; i++ ) {

    read(fd, currentArea->name, CHARS_PER_AREA_LABEL);
    read(fd, &(currentArea->fract), sizeof(double));

    if (i+1<n) {
      currentArea->next = (Area *) calloc(1, sizeof(Area));
      MCHECK(currentArea->next);
      currentArea = currentArea->next;
    } else {
      currentArea->next = NULL;
    }
  }
  return(return_list);
}

convert_object_pre2(Object *to_obj, Object_pre2 *from_obj, int version_num, int update_sizes)
{
  int	i,n,shape;

  if (bigEndian)
    flip_object_to_big(from_obj, version_num);

  to_obj->header.type=from_obj->header.type;
  shape=to_obj->header.shape=from_obj->header.shape;

  if (shape==ELLIPSE) {
    copy_pos(&(to_obj->ellipse.upper_left),&(from_obj->ellipse.upper_left));
    copy_pos(&(to_obj->ellipse.size),&(from_obj->ellipse.size));
  }
  else {
    n=to_obj->polygon.num_points=from_obj->polygon.num_points;
    for (i=0; i<n; i++)
      copy_pos(&(to_obj->polygon.points[i]),&(from_obj->polygon.points[i]));
  }

  to_obj->header.strength = 2;
  to_obj->header.inj_num = 0;
  if (update_sizes)
    to_obj->header.size = object_size(to_obj);

  if (bigEndian)
    flip_object_to_little(to_obj, DATA_FILE_VERSION);
}


/*
 * read area names and corresponding objects and incorporate into area_list
 */
read_areas_file(filename, image_num)
    char	*filename;
    int	image_num;
{
  int		fd;
  AreaIn	area_in;
  Object	*obj;
  Object_pre2	*old_obj;
  AreaDef	*new_area, *existing_area;
  AreaDef	*check_new_area();

  if ((fd = open(filename, O_RDONLY))==-1)
    return(0);

  while(read(fd, &area_in, sizeof(AreaIn))==sizeof(AreaIn)) {
    obj=(Object *)calloc(1,sizeof(Object));
    MCHECK(obj);
    old_obj=(Object_pre2 *)calloc(1,sizeof(Object_pre2));
    MCHECK(old_obj);
    read(fd, old_obj, sizeof(Object_pre2));
    old_obj->header.shape=POLYGON;
    old_obj->header.type=SEARCH_AREA;
    if (bigEndian) {
      flip_int(&(old_obj->header.shape));
      flip_int(&(old_obj->header.type));
    }
    convert_object_pre2(obj, old_obj, 1.2, 0);
    if (bigEndian)
      flip_object_to_big(obj,DATA_FILE_VERSION);
    free(old_obj);

    if (existing_area=check_new_area(area_in.name, area_list)) {
      obj->header.next=existing_area->objects[image_num];
      existing_area->objects[image_num]=obj;
    }
    else {
      new_area=(AreaDef *)calloc(1,sizeof(AreaDef));
      MCHECK(new_area);
      strcpy(new_area->name, area_in.name);
      new_area->objects=(Object **)calloc(num_images,sizeof(Object *));
      MCHECK(new_area->objects);
      new_area->objects[image_num]=obj;
      new_area->next=area_list;
      area_list=new_area;
    }
  }
  close(fd);
  return(1);
}
