/*
 * routines for computing areal overlaps using X drawing functions
 *
 * Copyright 1994, 1995 Bill Press and Bruno Olshausen
 * Washington University School of Medicine
 *
 */


#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include "xanat.h"
#include "ui.h"
#include "ui_objects.h"

/* Bounding box routines */

compute_object_list_icon_bounding_box(obj,mode,return_box)
    Object	*obj;
    int	mode;
    Box	*return_box;
{
  Box	obj_box;

  return_box->ul.x=icon_width_max;
  return_box->ul.y=icon_height_max;
  return_box->lr.x=return_box->lr.y=0;
  while (obj) {
    if (obj->header.type==mode && (inject_toggles[obj->header.inj_num] ||
				   mode==SEARCH_AREA)) {
      compute_object_icon_bounding_box(obj,&obj_box);
      if (obj_box.ul.x<return_box->ul.x)
	return_box->ul.x=obj_box.ul.x;
      if (obj_box.ul.y<return_box->ul.y)
	return_box->ul.y=obj_box.ul.y;
      if (obj_box.lr.x>return_box->lr.x)
	return_box->lr.x=obj_box.lr.x;
      if (obj_box.lr.y>return_box->lr.y)
	return_box->lr.y=obj_box.lr.y;
    }
    obj=obj->header.next;
  }
}

compute_object_icon_bounding_box(obj,box)
    Object	*obj;
    Box	*box;
{
  if (obj->header.shape==ELLIPSE)
    compute_ellipse_icon_bounding_box(obj,box);
  else
    compute_polygon_icon_bounding_box(obj,box);
}

objects_within_bounding_box_p(object_list,box)
    Object	*object_list;
    Box	*box;
{
  int		inside_p;
  Object	*obj;
  Box		obj_box;

  inside_p=0;
  obj=object_list;
  while (obj) {
    compute_object_icon_bounding_box(obj,&obj_box);
    if (!((obj_box.lr.x < box->ul.x) || (obj_box.lr.y < box->ul.y) ||
	  (obj_box.ul.x > box->lr.x) || (obj_box.ul.y > box->lr.y))) {
      inside_p++;
      break;
    }
    obj=obj->header.next;
  }
  return(inside_p);
}

in_box_p(x,y,box)
    int	x,y;
    Box	*box;
{
  if (x>box->ul.x && x<box->lr.x && y>box->ul.y && y<box->lr.y)
      return(1);
  else
      return(0);
}

/* Drawing into pixmaps */

clear_icon_pixmap()
{
  XSetForeground(display,pixmap_gc,0);
  XFillRectangle(display,analyze_pixmap,pixmap_gc,0,0,
		 icon_width_max,icon_height_max);
}

draw_object_in_icon_pixmap(Object *obj, int fill)
{
  clear_icon_pixmap();
  if (obj->header.shape==ELLIPSE)
    draw_ellipse_in_icon_pixmap(obj, fill);
  else
    draw_polygon_in_icon_pixmap(obj, fill);
}


draw_object_list_in_icon_pixmap(Object *object_list, int mode,
				int inj, int fill, int erase_first)
{
  Object	*obj;
  int		strengths;

  if (erase_first)
    clear_icon_pixmap();

  for (strengths=0; strengths<NUM_STRENGTHS; strengths++) {

    obj=object_list;

    while(obj) {
      if ((obj->header.strength==strengths) &&
	  (obj->header.type==mode) && 
	  (obj->header.inj_num==inj || inj==NUM_DIFF_INJ) &&
	  (inject_toggles[obj->header.inj_num] || mode==SEARCH_AREA))
	if (obj->header.shape==ELLIPSE)
	  draw_ellipse_in_icon_pixmap(obj, fill);
	else
	  draw_polygon_in_icon_pixmap(obj, fill);

      obj=obj->header.next;
    }
  }
}

/* Reading from pixmaps */

get_search_ximage()
{
  XDestroyImage(search_ximage);
  search_ximage=XGetImage(display, analyze_pixmap,
			  0,0,icon_width_max,icon_height_max,
			  AllPlanes,ZPixmap);
}

get_data_ximage()
{
  XDestroyImage(data_ximage);
  data_ximage=XGetImage(display, analyze_pixmap,
			0,0,icon_width_max,icon_height_max,
			AllPlanes,ZPixmap);
}

search_ximage_pixel(int i, int j)
{
  return(XGetPixel(search_ximage, i,j));
}


data_ximage_pixel(int i, int j)
{
  return(XGetPixel(data_ximage, i,j));
}


/* region overlap computation */

region_overlap(Box *box)
{
  int		i,j,overlap;

  overlap=0;
  for (j=box->ul.y; j<=box->lr.y; j++) {
    for (i=box->ul.x; i<=box->lr.x; i++) {
      if (data_ximage_pixel(i,j) && search_ximage_pixel(i,j))
	overlap++;
    }
  }
  return(overlap);
}


region_weighted_overlap(Box *box)
{
  int		i,j,overlap,pix_strength;

  overlap=0;
  for (j=box->ul.y; j<=box->lr.y; j++) {
    for (i=box->ul.x; i<=box->lr.x; i++) {
      if ((pix_strength=data_ximage_pixel(i,j)) && search_ximage_pixel(i,j))
	overlap+=pix_strength;
    }
  }
  return(overlap);
}


/* min and max routines */

double min_val(double min1, double min2)
{
  return((min1<min2) ? min1 : min2);
}

double max_val(double max1, double max2)
{
  return((max1>max2) ? max1 : max2);
}

double find_zero_point(double min, double max)
{
  if (min<0)
    return (max_val(fabs(min),fabs(max))/2);
  else
    return (0);
}

find_minmax(double *min, double *max, double **heatMap)
{
  int		plane, i;    
  double	*heatMapPointer;

  *min = *max = **heatMap;

  for ( plane=0; plane<num_images; plane++ ) {

    heatMapPointer = heatMap[plane];

    for (i=0; i<icon[plane].nrc; i++) {

      if (*heatMapPointer < *min)
	*min = *heatMapPointer;
	    
      if (*heatMapPointer > *max)
	*max = *heatMapPointer;

      heatMapPointer++;
    }
  }
}

evenout_minmax(double *min, double *max)
{
  if (*min<0) {
    if (*max > -*min)
      *min = -*max;
    else
      *max = -*min;
    *max *= 2.0;
  }
}

/**********  Comparison routines  **********/


do_heatmap_comparison(Datum *from1, Datum *from2, Datum *to_datum, int action)
{
  double	*from1_ptr, *from2_ptr, *to_ptr;
  double	min, max;
  int		plane, i;

  for ( plane=0; plane<num_images; plane++ ) {

    from1_ptr = from1->heatmap[plane];
    from2_ptr = from2->heatmap[plane];
    to_ptr = to_datum->heatmap[plane];

    for ( i=0; i<icon[plane].nrc; i++ )
      switch (action) {
       case SUBTRACTION:
	*to_ptr++ = *from2_ptr++ - *from1_ptr++;
	break;
       case ADDITION:
	*to_ptr++ = *from2_ptr++ + *from1_ptr++;
	break;
       case MULTIPLICATION:
	*to_ptr++ = *from2_ptr++ * *from1_ptr++;
	break;
      }
  }

  find_minmax(&min, &max, to_datum->heatmap);

  if (action!=SUBTRACTION) {

    to_datum->heatmap_min = !min ? 0 :
      min_val(min, min_val(from1->heatmap_min, from2->heatmap_min));

    to_datum->heatmap_max =
      max_val(max, max_val(from1->heatmap_max, from2->heatmap_max));
  }
  else {

    to_datum->heatmap_min = !min ? 0 :
      min_val(min, min_val(-from1->heatmap_max, from2->heatmap_min));

    to_datum->heatmap_max =
      max_val(max, max_val(-from1->heatmap_min, from2->heatmap_max));
  }

  to_datum->analysis_midpoint = find_zero_point(to_datum->heatmap_min,
						to_datum->heatmap_max);
}


add_data_into_heatmap(Datum *from1, Datum *from2, Datum *to_datum)
{
  Datum		*anl_datum, *obj_datum;

  double	*from_ptr;
  double	*to_ptr;

  double	min;
  double	max;

  int		plane;
  int		i,j;

  if (from1->type==ANALYSIS_TYPE) {
    anl_datum=from1;
    obj_datum=from2;
  }
  else {
    anl_datum=from2;
    obj_datum=from1;
  }

  find_minmax(&min, &max, anl_datum->heatmap);

  for (plane=0; plane<num_images; plane++) {

    draw_object_list_in_icon_pixmap(obj_datum->objects[plane],
				    INJECTION,NUM_DIFF_INJ,0,1);
    draw_object_list_in_icon_pixmap(obj_datum->objects[plane],
				    LABEL,NUM_DIFF_INJ,0,0);
    get_data_ximage();

    from_ptr=anl_datum->heatmap[plane];
    to_ptr=to_datum->heatmap[plane];

    for (j=0; j<icon[plane].height; j++)
      for (i=0; i<icon[plane].width; i++)

	if (data_ximage_pixel(i,j))
	  if (*from_ptr++ < min+(max-min)*(NUM_COLORS-1)/NUM_COLORS)
	    *to_ptr++ = max;
	  else
	    *to_ptr++ = min;
	else
	  *to_ptr++ = *from_ptr++;
  }
  to_datum->heatmap_min = anl_datum->heatmap_min;
  to_datum->heatmap_max = anl_datum->heatmap_max;
  to_datum->analysis_midpoint = anl_datum->analysis_midpoint;
}


/* 
 * analysis image composite
 */

static double f[4]={0.0, 0.25, 0.5, 0.75};

interpolate_analyzed_image(double *image_data, int image_width,
			   int image_height, double *icon_data,
			   int icon_width, int icon_height)
{
  int		i,j,m,n;
  double	*I0,*I1,y1,y2;

  for (j=0; j<icon_height-1; j++) {
    for (n=0; n<ICON_SF; n++) {
      I0=icon_data+j*icon_width;
      I1=image_data+(j*ICON_SF+n)*image_width;
      for (i=0; i<icon_width-1; i++) {
	for (m=0; m<ICON_SF; m++) {
	  y1=(1-f[m])*(*I0) + f[m]*(*(I0+1));
	  y2=(1-f[m])*(*(I0+icon_width)) + f[m]*(*(I0+icon_width+1));
	  *I1++=(1-f[n])*y1 + f[n]*y2;
	}
	I0++;
      }
    }
  }
}

/* Renormalizing heatmap */


rescaleHeatMap(double **heatMap, double min, double max)
{
  int		plane, i;    
  double	*heatMapPointer;

  for ( plane=0; plane<num_images; plane++ ) {

    heatMapPointer = heatMap[plane];

    for (i=0; i<icon[plane].nrc; i++) {

      if (min<0)
	*heatMapPointer += -min;

      if (max!=0)
	*heatMapPointer /= max;	    
	    
      heatMapPointer++;
    }
  }
}


compute_composite_analysis_image(byte *image_data, double *analyzed_image,
				 byte *composite, double min,
				 double max, int nrc)
{
  register int		i;
  register double	*aptr;
  register byte		*cptr,*iptr,p;
  register double	norm_heatval;

  evenout_minmax(&min, &max);  

  aptr=analyzed_image;
  iptr=image_data;
  cptr=composite;

  for (i=0; i<nrc; i++) {

    /* Normalize heatmap to [min,max] */

    norm_heatval = *aptr++;

    if (min<0)
      norm_heatval += -min;

    if (max!=0)
      norm_heatval /= max;	    

    /* Round 0.5 down to green */

    p = (byte) ceil ((double)norm_heatval * NUM_COLORS);
    if (p)
      p--;

    /* Scale to heatmap values */

    *cptr++ = HEATMAP_OFFSET + p*NUM_GREY_LEVELS_PER_COLOR +
      (*iptr++ * NUM_GREY_LEVELS_PER_COLOR)/NUM_GREY_LEVELS;
  }
}

