/* 
 * routines for keeping track of the colormap 
 *
 * Copyright 1994, 1995 Bill Press and Bruno Olshausen
 * Washington University School of Medicine
 *
 */

#include <stdio.h>
#include <math.h>

#include <X11/cursorfont.h>

#include "xanat.h"
#include "ui.h"


static int inj_label_colors[NUM_DIFF_INJ][2][3];

int get_object_color(int type, int inj_num)
{
  if (type==SEARCH_AREA)
    return(SEARCH_INDEX);
  else
    return(GREYS_OFFSET+inj_num*2+type);
}

set_colormap_data()
{
  int		i,j;
  int		index, shift;
  XColor	color[NUM_GREY_LEVELS+INJ_LABEL_SIZE+2];

  shift=8+2;
  for (i=0; i<NUM_GREY_LEVELS; i++) {
    color[i].pixel = i;
    color[i].red   = i<<shift;
    color[i].green = i<<shift;
    color[i].blue  = i<<shift;
    color[i].flags = DoRed | DoGreen | DoBlue;
  }
  XStoreColors(display,cmap,color,NUM_GREY_LEVELS);

  for (i=0, index=NUM_GREY_LEVELS; i<NUM_DIFF_INJ; i++)
    for (j=0; j<2; j++) {
      color[index].pixel = GREYS_OFFSET+i*2+j;
      color[index].red   = 256*(255-(255-inj_label_colors[i][j][0]));
      color[index].green = 256*(255-(255-inj_label_colors[i][j][1]));
      color[index].blue  = 256*(255-(255-inj_label_colors[i][j][2]));
      color[index].flags = DoRed | DoGreen | DoBlue;
      index++;
    }

  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE].pixel = BLUE;
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE].red = 0;
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE].green = 0;
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE].blue = 256*255;
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE].flags = DoRed | DoGreen | DoBlue;

  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE+1].pixel = DARK_BLUE;
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE+1].red =
    (256*(NUM_GREY_LEVELS_PER_COLOR/2)*
     my_red(1))/(NUM_GREY_LEVELS_PER_COLOR-1);
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE+1].green =
    (256*(NUM_GREY_LEVELS_PER_COLOR/2)*
     my_green(1))/(NUM_GREY_LEVELS_PER_COLOR-1);
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE+1].blue =
    (256*(NUM_GREY_LEVELS_PER_COLOR/2)*
     my_blue(1))/(NUM_GREY_LEVELS_PER_COLOR-1);
  color[NUM_GREY_LEVELS+INJ_LABEL_SIZE+1].flags = DoRed | DoGreen | DoBlue;

  XStoreColors(display,cmap,color,NUM_GREY_LEVELS+INJ_LABEL_SIZE+2);
}

set_colormap_object_color(int mode)
{
  int		i;
  XColor 	color[NUM_GREY_LEVELS];

  for (i=0; i<NUM_GREY_LEVELS; i++) {
    color[i].pixel = NUM_GREY_LEVELS+i;
    color[i].red   = 256*(255-(255-inj_label_colors[inject_num][mode][0]));
    color[i].green = 256*(255-(255-inj_label_colors[inject_num][mode][1]));
    color[i].blue  = 256*(255-(255-inj_label_colors[inject_num][mode][2]));
    color[i].flags = DoRed | DoGreen | DoBlue;
  }
  XStoreColors(display,cmap,color,NUM_GREY_LEVELS);
}

set_colormap_heatmap()
{
  int		i, j, index;
  XColor	color[NUM_COLORS*NUM_GREY_LEVELS_PER_COLOR];

  for (j=1, index=0; j<=NUM_COLORS; j++) {
    for (i=0; i<NUM_GREY_LEVELS_PER_COLOR; i++) {
      color[index].pixel = HEATMAP_OFFSET+index;
      color[index].red   = (256*i*my_red(j))/(NUM_GREY_LEVELS_PER_COLOR-1);
      color[index].green = (256*i*my_green(j))/(NUM_GREY_LEVELS_PER_COLOR-1);
      color[index].blue  = (256*i*my_blue(j))/(NUM_GREY_LEVELS_PER_COLOR-1);
      color[index].flags = DoRed | DoGreen | DoBlue;
      index++;
    }
  }
  XStoreColors(display,cmap,color,NUM_COLORS*NUM_GREY_LEVELS_PER_COLOR);
}

set_cursor_colors()
{
  int	i;
  int	mode_color_num;

  for (i=0; i<3; i++) {
    if (i==SEARCH_AREA) {
      cursor_color[i].red   = 256*my_red(SEARCH_COLOR);
      cursor_color[i].green = 256*my_green(SEARCH_COLOR);
      cursor_color[i].blue  = 256*my_blue(SEARCH_COLOR);
    }
    else {
      cursor_color[i].red   = 256*inj_label_colors[inject_num][i][0];
      cursor_color[i].green = 256*inj_label_colors[inject_num][i][1];
      cursor_color[i].blue  = 256*inj_label_colors[inject_num][i][2];
    }
    cursor_color[i].flags = DoRed | DoGreen | DoBlue;
  }

  cursor_white.pixel=WHITE;
  XQueryColor(display,cmap,&cursor_white);

  cursor_black.pixel=BLACK;
  XQueryColor(display,cmap,&cursor_black);
}  

set_image_win_colors()
{
  if (selected_datum[view_mode]) {
    if (!is_analysis)
      set_colormap_object_color(object_mode);
    set_cursor_colors();
    XRecolorCursor(display,arrow_cursor,&cursor_color[object_mode],
		   &cursor_white);
    XRecolorCursor(display,cross_cursor,&cursor_color[object_mode],
		   &cursor_white);
  }
  else
    XRecolorCursor(display,arrow_cursor,&cursor_black,&cursor_white);
}

set_xor_stipple()
{
  XSetStipple(display, image_xor_stip_gc, image_stipple_pixmap[strength_num]);
  if (strength_num!=(NUM_STRENGTHS-1)) {
    XSetLineAttributes(display, image_xor_dash_gc, 0,
		       LineOnOffDash, CapButt, JoinMiter);
    XSetDashes(display, image_xor_dash_gc, 0, dashes[strength_num], 2);
  }
  else
    XSetLineAttributes(display, image_xor_dash_gc, 0,
		       LineSolid, CapButt, JoinMiter);
}


set_xor_mask()
{
  static unsigned long heatmap_xor_val = (256-HEATMAP_OFFSET)/2;

  if (heatmap_xor_val!=NUM_GREY_LEVELS)
    if (is_analysis) {
      XSetPlaneMask(display, image_xor_stip_gc, heatmap_xor_val);
      XSetPlaneMask(display, image_xor_dash_gc, heatmap_xor_val);
      XSetPlaneMask(display, image_handle_gc, heatmap_xor_val);
    }
    else {
      XSetPlaneMask(display, image_xor_stip_gc, NUM_GREY_LEVELS);
      XSetPlaneMask(display, image_xor_dash_gc, NUM_GREY_LEVELS);
      XSetPlaneMask(display, image_handle_gc, NUM_GREY_LEVELS);
    }
}


/*
 * Converting from hsv (0-360, 0-1, 0-1) to rgb (0-255, 0-255, 0-255)
 */

int scaled (double to_scale)
{
  return ((int) rint(to_scale*255.0));
}

convert_hsv_to_rgb (double h, double s, double v,
		    int *r, int *g, int *b)
{
  int i;
  double f, p, q, t;

  if (!s) {
    *r = scaled (v);
    *g = scaled (v);
    *b = scaled (v);
  }
  else {
    h = 360.0-h;
    if (h==360.0)
      h = 0.0;
    h /= 60.0;
    i = (int) floor(h);
    f = h-i;
    p = v*(1-s);
    q = v*(1-s*f);
    t = v*(1-s*(1-f));
    switch (i) {
     case 0:
      *r = scaled (v);
      *g = scaled (t);
      *b = scaled (p);
      break;
     case 1:
      *r = scaled (q);
      *g = scaled (v);
      *b = scaled (p);
      break;
     case 2:
      *r = scaled (p);
      *g = scaled (v);
      *b = scaled (t);
      break;
     case 3:
      *r = scaled (p);
      *g = scaled (q);
      *b = scaled (v);
      break;
     case 4:
      *r = scaled (t);
      *g = scaled (p);
      *b = scaled (v);
      break;
     case 5:
      *r = scaled (v);
      *g = scaled (p);
      *b = scaled (q);
      break;
    }
  }
}

set_inj_label_colors ()
{
  int i;
  double hue_space = 360.0/(NUM_DIFF_INJ*3);

  for (i=0; i<NUM_DIFF_INJ; i++) {
    convert_hsv_to_rgb(hue_space*(i*3),1.0,1.0,
		       &inj_label_colors[i][0][0],
		       &inj_label_colors[i][0][1],
		       &inj_label_colors[i][0][2]);
    convert_hsv_to_rgb(hue_space*(i*3+1),1.0,1.0,
		       &inj_label_colors[i][1][0],
		       &inj_label_colors[i][1][1],
		       &inj_label_colors[i][1][2]);
  }
}

/*
 * color scheme used by Matt Wilson's xview program
 */
#define WI ((double)NUM_COLORS/2)
#define	arc(x)	(255.0*sqrt(1.0-(((x)-WI)/WI)*(((x)-WI)/WI)))
#define	carc(x)	(((0<=(x))&&((x)<= 2*WI ))?arc(x):0)

init_colorstrip(byte *data)
{
  int	i,j,color,p;
  byte	*dptr;

  for (i=0; i<colorstrip_width; i++) {
    color=(i*NUM_COLORS)/colorstrip_width;
    p=HEATMAP_OFFSET+(color+1)*NUM_GREY_LEVELS_PER_COLOR-1;
    dptr=data;
    for (j=0; j<colorstrip_height; j++) {
      *dptr = p;
      dptr+=colorstrip_width;
    }
    data++;
  }
}

my_red(int i)
{
  return((int)carc((double)(i-WI)));
}

my_green(int i)
{
  return((int)carc((double)(i)));
}

my_blue(int i)
{
  return((int)carc((double)i+WI));
}
