/* 
 * routines for drawing (except for ellipses and polygons)
 *
 * Copyright 1994, 1995 Bruno Olshausen and Bill Press
 * Washington University School of Medicine
 *
 */

#include <stdio.h>
#include <string.h>

#include "xanat.h"
#include "ui.h"
#include "ui_objects.h"
#include "ui_text.h"


refresh_data_displays()
{
  check_switch_modes();

  set_image_win_colors();

  redraw_icons_win();
  redraw_image_win(1);
  redraw_data_win();
  redraw_index_win();
  if (!wrote_message)
    redraw_message_win();
}

/* Icon window drawing routines */

redraw_icon(image_num)
    int	image_num;
{
  int		x,y,i;
  Object	*obj;
  Datum		*datum;
  XPoint	points[3];

  byte		*icon_composite=(byte *) calloc(icon_nrc_max, sizeof(byte));
  extern byte	*get_image();
  XImage  	*icon_ximage;

  XRectangle	clip_rectangle;

  icon_ximage = XCreateImage(display,visual,8,ZPixmap,0,icon[image_num].data,
			     icon[image_num].width,icon[image_num].height,8,0);

  x=icon_pos[image_num].x;
  y=icon_pos[image_num].y;

  clip_rectangle.x = clip_rectangle.y = 0;
  clip_rectangle.width = icon[image_num].width;
  clip_rectangle.height = icon[image_num].height;

  XSetClipRectangles(display,icons_win.gc,x,y,&clip_rectangle,1,Unsorted);
  XSetClipRectangles(display,icons_stip_gc,x,y,&clip_rectangle,1,Unsorted);

  datum=selected_datum[view_mode];

  if (is_analysis) {
    compute_composite_analysis_image(get_image(&icon[image_num]),
				     datum->heatmap[image_num], icon_composite,
				     datum->heatmap_min, datum->heatmap_max,
				     icon[image_num].nrc);
    icon_ximage->data=(char *)icon_composite;
  }
  else
    icon_ximage->data=(char *)get_image(&icon[image_num]);

  XPutImage(display,icons_win.window,icons_win.gc,icon_ximage,
	    0,0,x,y, icon[image_num].width, icon[image_num].height);
  
  if (datum)
    if (datum->validSlice[image_num]) {
      obj=datum->objects[image_num];
      while(obj) {
	draw_object_in_icon(obj,image_num);
	obj=obj->header.next;
      }
    } else {
      for (i=0; i<3; i++) {
	points[i].x=x+valid_triangle[i].x/ICON_SF;
	points[i].y=y+valid_triangle[i].y/ICON_SF;
      }
      XSetForeground(display,icons_win.gc,DARK_BLUE);
      XFillPolygon(display,icons_win.window,icons_win.gc,
		   points, 3, Convex, CoordModeOrigin);
    }

  XFlush(display);

  icon_ximage->data = NULL;
  XDestroyImage(icon_ximage);
  free(icon_composite);
}

redraw_icons_win()
{
  int	i;

  for (i=0; i<num_images; i++)
    redraw_icon(i);
}

/* Image window drawing routines */

redraw_image_win(recompute_p)
    int	recompute_p;
{
  Object	*obj;
  Datum		*datum;

  double	*heatmap_blowup=(double *)calloc(image_nrc_max,sizeof(double));
  byte		*image_composite=(byte *)calloc(image_nrc_max,sizeof(byte));

  XImage  	*ximage;
  extern byte	*get_image();

  ximage = XCreateImage(display, visual, 8, ZPixmap, 0,
			image[current_image].data, image[current_image].width,
			image[current_image].height, 8, 0);
    
  datum = selected_datum[view_mode];

  XClearWindow(display,image_win.window);
  clear_handle_var();

  if (is_analysis) {
    if (recompute_p) {
      interpolate_analyzed_image(heatmap_blowup,
				 image[current_image].width,
				 image[current_image].height,
				 datum->heatmap[current_image],
				 icon[current_image].width,
				 icon[current_image].height);
      compute_composite_analysis_image(get_image(&image[current_image]),
				       heatmap_blowup, image_composite,
				       datum->heatmap_min, datum->heatmap_max,
				       image[current_image].nrc);
    }
    ximage->data=(char *)image_composite;
    draw_colorstrip();
  }
  else {
    ximage->data=(char *)get_image(&image[current_image]);
    draw_blankstrip();
  }

  XPutImage(display,image_win.window,image_win.gc,ximage,0,0,0,0,
	    image[current_image].width,image[current_image].height);

  if (datum)
    if (datum->validSlice[current_image]) {
      obj=datum->objects[current_image];
      while(obj) {
	draw_object(obj);
	if (obj==selected_object)
	  draw_handle_bars();
	obj=obj->header.next;
      }
    }
    else {
      XSetForeground(display,image_win.gc, DARK_BLUE);
      XFillPolygon(display,image_win.window,image_win.gc,
		   valid_triangle, 3, Convex, CoordModeOrigin);
    }

  ximage->data = NULL;
  XDestroyImage(ximage);
  free(heatmap_blowup);
  free(image_composite);
}

reset_image_win()
{
  set_image_win_colors();
  set_xor_stipple();
  redraw_image_win(0);
}

draw_blankstrip()
{
  XClearArea(display,image_win.window, 0,image_height_max,
	     colorstrip_width,colorstrip_height, False);
  XSetForeground(display,image_win.gc,BLACK);
  XDrawLine(display,image_win.window,image_win.gc,
	    0,image_height_max,colorstrip_width,image_height_max);
}

draw_colorstrip()
{
  int	bar_pos;

  XPutImage(display,image_win.window,image_win.gc,colorstrip_ximage,0,0,
	    0,image_height_max, colorstrip_width,colorstrip_height);
  bar_pos=colorstrip_width*selected_datum[view_mode]->analysis_midpoint;
  XSetForeground(display,image_win.gc,BLACK);
  XFillRectangle(display,image_win.window,image_win.gc,
		 bar_pos, image_height_max,  3,colorstrip_height);
  XDrawLine(display,image_win.window,image_win.gc,
	    0,image_height_max,colorstrip_width,image_height_max);
}

/* Control window drawing routines */

str_pix_len(string,font)
    char	*string;
    font_obj *font;
{
  return(XTextWidth(font->info,string,strlen(string)));
}

set_control_cols(control_struct *cptr)
{
  if (view_mode==SEARCH && cptr->id==LIST_ID)
    XSetForeground(display,control_win.gc,GREY);
}

redraw_control_win()
{
  int	i,j,index,x,y;
	
  XClearWindow(display,control_win.window);

  index=0;
  for (j=0; j<num_control_rows; j++) {
    for (i=0; i<num_controls_per_row && index<NUM_CONTROLS; i++) {
      x= control_spacing.x*i;
      y= control_spacing.y*j;
      XSetForeground(display,control_win.gc,BLACK);
      XDrawLine(display,control_win.window,control_win.gc,
		x+control_spacing.x,y,
		x+control_spacing.x,y+control_spacing.y);
      if (control_list[is_stack][is_analysis][index].num_sub_choices)
	draw_submenu(&control_list[is_stack][is_analysis][index],x,y);
      if (is_analysis)
	XSetForeground(display,control_win.gc,RED);
      else
	XSetForeground(display,control_win.gc,BLUE);
      set_control_cols(&control_list[is_stack][is_analysis][index]);
      XDrawImageString(display,control_win.window,control_win.gc,
		       x+(control_spacing.x-str_pix_len(control_list[is_stack][is_analysis][index].label,&medium_font))/2,
		       y+control_spacing.y/2,
		       control_list[is_stack][is_analysis][index].label,
		       strlen(control_list[is_stack][is_analysis][index].label));
      index++;
    }
    if (j>0) {
      XSetLineAttributes(display,control_win.gc,2,LineSolid,CapButt,JoinMiter);
      XSetForeground(display,control_win.gc,BLACK);
      XDrawLine(display,control_win.window,control_win.gc,
		0,y,control_win.width,y);
      XSetLineAttributes(display,control_win.gc,1,LineSolid,CapButt,JoinMiter);
    }
  }
}

set_submenu_greys(control_struct *cptr, int i)
{
  if (view_mode==SEARCH && cptr->id==LIST_ID)
    XSetForeground(display,control_win.gc,GREY);
  if (view_mode!=STACK)
    if (cptr->id==STACK_ID && !strcmp(cptr->sub_label[i],"remove"))
      XSetForeground(display,control_win.gc,GREY);
  if (is_analysis && cptr->id==DRAW_MODE_ID &&
      !strcmp(cptr->sub_label[i],"rev"))
    XSetForeground(display,control_win.gc,GREY);
}

set_submenu_selects(control_struct *cptr, int i)
{
  extern int draw_mode, view_mode, image_mode, object_mode;
  extern int get_object_color();

  if ((cptr->id==DRAW_MODE_ID && draw_mode==i) ||
      (cptr->id==VIEW_MODE_ID && view_mode==i) ||
      (cptr->id==IMAGE_MODE_ID && image_mode==i) ||
      (cptr->id==INJ_TYPE_ID && inject_num==i && !is_analysis) ||
      (cptr->id==WEIGHT_ID && strength_num==i) ||
      (cptr->id==DRAW_MODE_ID && !strcmp(cptr->sub_label[i],"fill") &&
       fill_mode))
    XSetForeground(display,control_win.gc,RED);

  if (cptr->id==ANALYSIS_METHOD_ID && analysis_type==i)
    XSetForeground(display,control_win.gc,RED);

  if (cptr->id==INJ_OR_LABEL_ID && object_mode==i)
    if (object_mode==INJECTION)
      XSetForeground(display,control_win.gc,
		     get_object_color(INJECTION,inject_num));
    else
      XSetForeground(display,control_win.gc,
		     get_object_color(LABEL,inject_num));
}

set_submenu_whites(control_struct *cptr, int i)
{
  if (cptr->id==INJ_TYPE_ID && !inject_toggles[i])
      XSetForeground(display,control_win.gc,WHITE);
}

char *get_submenu_string (control_struct *cptr, int i)
{
  static char inj_or_weight_num[2];

  if (cptr->id==INJ_TYPE_ID || cptr->id==WEIGHT_ID) {
    sprintf(inj_or_weight_num, "%2d", i+1);
    return(inj_or_weight_num);
  }
  else
    return(cptr->sub_label[i]);
}

draw_submenu(cptr,x,y)
    control_struct	*cptr;
    int		x,y;
{
  int	i,sub_hspace;
  char *submenu_string;
  char inj_or_weight_num[2];
  int inj_or_weight_color;

  sub_hspace=control_spacing.x/cptr->num_sub_choices;
  XSetFont(display,control_win.gc,small_font.info->fid);
  for (i=0; i<cptr->num_sub_choices; i++) {
    if (i!=0) {
      XSetForeground(display,control_win.gc,GREY);
      XDrawLine(display,control_win.window,control_win.gc,
		x+i*sub_hspace, y, x+i*sub_hspace, y+control_spacing.y);
    }
    XSetForeground(display,control_win.gc,BLACK);
    set_submenu_greys(cptr,i);
    set_submenu_selects(cptr,i);
    set_submenu_whites(cptr,i);
    submenu_string = get_submenu_string(cptr,i);
    XDrawString(display,control_win.window,control_win.gc,
		x+(int)((sub_hspace)*(0.5+i))-
		str_pix_len(submenu_string,&small_font)/2,
		y+control_spacing.y-3,
		submenu_string,strlen(submenu_string));
    XSetForeground(display,control_win.gc,BLACK);
  }
  XSetFont(display,control_win.gc,medium_font.info->fid);
}


/* text windows */

draw_label(obj,label,x,y)
    win_obj	*obj;
    char	*label;
    int	x,y;
{
  XDrawString(display,obj->window,obj->gc,x,y,label,strlen(label));
  XDrawLine(display,obj->window,obj->gc,
	    x,y+2,x+str_pix_len(label,&medium_font),y+2);
}

draw_field(obj,string,x,y)
    win_obj	*obj;
    char	*string;
    int	x,y;
{
  char	*str;
  int	i, chars_per_line, chars_per_this_line, indent;

  str=string;
  i=0;
  indent=str_pix_len(INDENT,&medium_font);
  chars_per_line=(obj->width-(x+indent))/medium_font.width;
  while (strlen(str)>chars_per_line) {
    chars_per_this_line=chars_per_line;
    while (str[chars_per_this_line-1]!=' ' && chars_per_this_line>1)
      chars_per_this_line--;
    XDrawString(display,obj->window,obj->gc,
		x+indent, y+medium_font.height+BUFF + i*medium_font.height,
		str,chars_per_this_line);
    str+=chars_per_this_line;
    i++;
  }
  XDrawString(display,obj->window,obj->gc,
	      x+indent, y+medium_font.height+BUFF + i*medium_font.height,
	      str,strlen(str));
}

/* Data window routines */

static char	*data_label[NUM_DATA_ITEMS] = {"REFERENCE:",
						 "INJECTION AREAS:",
						 "LABEL AREAS:",
						 "CONFIDENCE:",
						 "LABEL TYPE (a/r):",
						 "COMMENTS:"};

static char	*data_title[NUM_MODES] = {"DATA",
					    "SEARCH",
					    "STACK",
					    "ANALYSIS"};

static char	*analysis_types[NUM_ANALYSES] = {"Superposition",
						   "Bayes",
						   "Subtraction",
						   "Addition",
						   "Multiplication"};

redraw_data_win()
{
  int	i,x,y;
  Datum	*datum;
	
  XClearWindow(display,data_win.window);
  XSetFont(display,data_win.gc,large_font.info->fid);
  XDrawString(display,data_win.window,data_win.gc,
	      (data_win.width-
	       str_pix_len(data_title[view_mode],&large_font))/2,
	      large_font.height+4,
	      data_title[view_mode],strlen(data_title[view_mode]));
  XSetFont(display,data_win.gc,medium_font.info->fid);

  if (datum=selected_datum[view_mode]) {
    x=FIELD_BEGIN_X;
    y=FIELD_BEGIN_Y;
    if (datum->type==DATA_TYPE) {
      for (i=0; i<NUM_DATA_ITEMS; i++) {
	draw_label(&data_win,data_label[i],x,y);
	datum2str(datum,i,string);
	draw_field(&data_win,string,x,y);
	y+=FIELD_STEP;
      }
    } else {
      i=0;
      draw_label(&data_win,data_label[i],x,y);
      datum2str(datum,i,string);
      draw_field(&data_win,string,x,y);
      i=1;
      y+=2*FIELD_STEP;
      draw_label(&data_win,"SEARCH AREAS:",x,y);
      datum2str(datum,i,string);
      draw_field(&data_win,string,x,y);

      y+=2*FIELD_STEP;
      draw_label(&data_win,"METHOD:",x,y);
      draw_field(&data_win,analysis_types[datum->analysisMethod],x,y);
    }
  }
}

/* Search window routines */

static char	*search_label[NUM_SEARCH_ITEMS]={"DIRECTION:",
						   "AREAS:",
						   "REFERENCES:",
						   "COMMENTS:"};
redraw_search_win()
{
  int	i,x,y;

  XClearWindow(display,search_win.window);
  XSetFont(display,search_win.gc,large_font.info->fid);
  XDrawString(display,search_win.window,search_win.gc,
	      (search_win.width-str_pix_len("SEARCH FIELDS",&large_font))/2,
	      large_font.height+4,
	      "SEARCH FIELDS",strlen("SEARCH FIELDS"));
  XSetFont(display,search_win.gc,medium_font.info->fid);

  draw_search_buttons();	      

  x=FIELD_BEGIN_X;
  y=FIELD_BEGIN_Y;
  for (i=0; i<NUM_SEARCH_ITEMS; i++) {
    draw_label(&search_win,search_label[i],x,y);
    search2str(&search_fields,i,string);
    draw_field(&search_win,string,x,y);
    y+=FIELD_STEP;
  }
}

draw_search_buttons()
{
  int	width,height;

  XSetLineAttributes(display,search_win.gc,2,LineSolid,CapButt,JoinMiter);
  width=search_box.lr.x-search_box.ul.x;
  height=search_box.lr.y-search_box.ul.y;
  XDrawRectangle(display,search_win.window,search_win.gc,
		 search_box.ul.x,search_box.ul.y, width,height);
  XDrawString(display,search_win.window,search_win.gc,
	      search_box.ul.x+(width-str_pix_len("search",&medium_font))/2,
	      search_box.ul.y+height-5,
	      "search",strlen("search"));

  width=analyze_box.lr.x-analyze_box.ul.x;
  height=analyze_box.lr.y-analyze_box.ul.y;
  XDrawRectangle(display,search_win.window,search_win.gc,
		 analyze_box.ul.x,analyze_box.ul.y, width,height);
  XDrawString(display,search_win.window,search_win.gc,
	      analyze_box.ul.x+(width-str_pix_len("analyze",&medium_font))/2,
	      analyze_box.ul.y+height-5,
	      "analyze",strlen("analyze"));
  XSetLineAttributes(display,search_win.gc,1,LineSolid,CapButt,JoinMiter);
}


/* index window routines */

#define ARROW_SIZE 8
#define ARROW_OFFSET 2

draw_arrow(y,sgn)
    int	y,sgn;
{
  int	xoffset,yoffset;

  xoffset=(INDEX_OFFSET_X-ARROW_SIZE)/2;
  yoffset=sgn*(INDEX_BOX_HEIGHT-ARROW_SIZE)/2;
  XDrawLine(display,index_win.window,index_win.gc,
	    xoffset,y+yoffset, xoffset+ARROW_SIZE/2,y+yoffset+sgn*ARROW_SIZE);
  XDrawLine(display,index_win.window,index_win.gc,
	    xoffset+ARROW_SIZE/2,y+yoffset+sgn*ARROW_SIZE,
	    xoffset+ARROW_SIZE,y+yoffset);
}

redraw_index_win()
{
  int	y,index_width,i;
  Datum	*dptr, *page_start;
  char	label[20];

  extern Datum *repage_index();

  XClearWindow(display,index_win.window);

  XSetFont(display,index_win.gc,large_font.info->fid);
  XDrawString(display,index_win.window,index_win.gc,
	    (index_win.width-str_pix_len(data_title[view_mode],&large_font))/2,
	      large_font.height+4,
	      data_title[view_mode],strlen(data_title[view_mode]));
  XSetFont(display,index_win.gc,medium_font.info->fid);

  XDrawRectangle(display,index_win.window,index_win.gc,
		 0, INDEX_OFFSET_Y, INDEX_OFFSET_X,
		 index_win.height-INDEX_OFFSET_Y);
  draw_arrow(index_stepup_box.lr.y, -1);
  draw_arrow(index_pageup_box.lr.y, -1);
  draw_arrow(index_pageup_box.lr.y-5, -1);
  draw_arrow(index_stepdown_box.ul.y, +1);
  draw_arrow(index_pagedown_box.ul.y, +1);
  draw_arrow(index_pagedown_box.ul.y+5, +1);
  
  page_start=repage_index();

  y=INDEX_OFFSET_Y;
  index_width=index_win.width-INDEX_OFFSET_X;
  dptr=page_start;
  for (i=0; i<num_datums_per_index_page && dptr; i++) {
    XDrawRectangle(display,index_win.window,index_win.gc,
		   INDEX_OFFSET_X ,y, index_width, INDEX_ICON_HEIGHT);
    if (dptr==selected_datum[view_mode])
      XDrawString(display,index_win.window,index_win.gc,
		  INDEX_OFFSET_X+2, y+INDEX_ICON_HEIGHT-5, ">" , 1);
    if (dptr->type==ANALYSIS_TYPE && dptr->analysisMethod>=COMPARE_OFFSET)
      sprintf(label,"%s %s (%d)", dptr->ref.abbrev,
	      dptr->ref.comparison, dptr->ref.num);
    else
      sprintf(label,"%s (%d)", dptr->ref.abbrev, dptr->ref.num);
    if (dptr->writeProtect)
      XSetForeground(display,index_win.gc,255);
    XDrawString(display,index_win.window,index_win.gc,
		INDEX_OFFSET_X+(index_width-str_pix_len(label,&medium_font))/2,
		y+INDEX_ICON_HEIGHT-5, label,strlen(label));
    y+=INDEX_ICON_HEIGHT;
    dptr=dptr->next[view_mode];
    XSetForeground(display,index_win.gc,BLACK);
  }
}


/* message window */

redraw_message_win()
{
  XClearWindow(display,message_win.window);

  XSetFont(display,message_win.gc,large_font.info->fid);
  XDrawString(display,message_win.window,message_win.gc,
	      (message_win.width-str_pix_len("MESSAGES",&large_font))/2,
	      large_font.height+4,
	      "MESSAGES",strlen("MESSAGES"));
  XSetFont(display,message_win.gc,medium_font.info->fid);
}


/* stereotaxic coordinate display */

redraw_coord_win()
{
  XClearWindow(display,coord_win.window);

  XSetFont(display,coord_win.gc,large_font.info->fid);
  XDrawString(display,coord_win.window,coord_win.gc,
	      (coord_win.width-str_pix_len("COORDINATES",&large_font))/2,
	      large_font.height+4,
	      "COORDINATES",strlen("COORDINATES"));
  XSetFont(display,coord_win.gc,medium_font.info->fid);
}

update_coord(x,y)
    int	x,y;
{
  StereotaxPoint	st_coord;
  char			coord_string[20];
  extern StereotaxPoint	win2coord();

  clear_coord();
  
  st_coord=win2coord(x,y,current_image,reverse_mode);
  sprintf(coord_string,"(%5.2lf, %5.2lf)",st_coord.x,st_coord.y);
  XDrawString(display,coord_win.window,coord_win.gc,
	      (coord_win.width-str_pix_len(coord_string,&medium_font))/2,
	      coord_win.height-5,
	      coord_string,strlen(coord_string));
}

clear_coord()
{
  XClearArea(display,coord_win.window,
	     0, coord_win.height-(medium_font.height+5),
	     coord_win.width, medium_font.height+5, False);
}


/* image reversal */

reverse_everything()
{
  int	i;

  for (i=0; i<num_images; i++) {
    reverse_image(image[i].data, image[i].width, image[i].height);
    reverse_image(icon[i].data, icon[i].width, icon[i].height);
    if (image[i].overlay) {
      reverse_image(image[i].overlay, image[i].width, image[i].height);
      reverse_image(icon[i].overlay, icon[i].width, icon[i].height);
    }
  }

  deselect_object();
  redraw_icons_win();
  redraw_image_win(0);
}

reverse_image(array,width,height)
    byte	*array;
    int	width,height;
{
  register int	i,j,halfway;
  register byte	*begin,*end,temp;

  halfway=width/2;
  for (j=0; j<height; j++) {
    begin=array;
    end=begin+width-1;
    for (i=0; i<halfway; i++) {
      temp=*begin;
      *begin++=*end;
      *end--=temp;
    }
    array+=width;
  }
}

reverse_analysis(array,width,height)
    double	*array;
    int	width,height;
{
  register int		i,j,halfway;
  register double	*begin,*end,temp;

  halfway=width/2;
  for (j=0; j<height; j++) {
    begin=array;
    end=begin+width-1;
    for (i=0; i<halfway; i++) {
      temp=*begin;
      *begin++=*end;
      *end--=temp;
    }
    array+=width;
  }
}

