/*
 * routines for ellipse drawing and manipulation
 *
 * Copyright 1994, 1995 Bruno Olshausen and Bill Press
 * Washington University School of Medicine
 *
 */

#include <stdio.h>
#include "xanat.h"
#include "ui.h"
#include "ui_objects.h"

#define UPPER_LEFT_HANDLE 0
#define UPPER_MIDDLE_HANDLE 1
#define UPPER_RIGHT_HANDLE 2
#define MIDDLE_RIGHT_HANDLE 3
#define LOWER_RIGHT_HANDLE 4
#define LOWER_MIDDLE_HANDLE 5
#define LOWER_LEFT_HANDLE 6
#define MIDDLE_LEFT_HANDLE 7


init_ellipse(ell,x,y)
    Ellipse	*ell;
    int	x,y;
{
  set_pos(&(ell->upper_left),x,y);
  set_pos(&(ell->size),0,0);
}

inc_ellipse_pos(ell,d)
    Ellipse	*ell;
    XPoint	*d;
{
  inc_pos(&(ell->upper_left),d);
}

int compare_ellipses (Ellipse *Ell1, Ellipse *Ell2)
{
  if (Ell1->upper_left.x == Ell2->upper_left.x &&
      Ell1->upper_left.y == Ell2->upper_left.y &&
      Ell1->size.x == Ell2->size.x && Ell1->size.y == Ell2->size.y)
    return (1);
  else
    return (0);
}

within_ellipse_p(ell,x,y)
    Ellipse	*ell;
    int	x,y;
{
  double  a, b;
  double  x0, y0;
  double  distance;

  x0 = (double) ell->upper_left.x + (double) ell->size.x / 2;
  y0 = (double) ell->upper_left.y + (double) ell->size.y / 2;

  a = (double) ell->size.x / 2;
  b = (double) ell->size.y / 2;

  distance = (x-x0)*(x-x0)/(a*a) + (y-y0)*(y-y0)/(b*b);
    
  if (distance<=1.0)
    return (1);
  else
    return (0);
}

compute_ellipse_icon_bounding_box(ell,box)
    Ellipse	*ell;
    Box	*box;
{
  PixPoint	ul,lr;

  ul.x=lr.x=ell->upper_left.x;
  ul.y=lr.y=ell->upper_left.y;
  inc_pos(&lr,&(ell->size));
  scale_pos(&box->ul,&ul,ICON_SF);
  scale_pos(&box->lr,&lr,ICON_SF);
}


process_ellipse_press(x,y)
    int	x,y;
{
  new_object(x,y);
  move_mode=STRETCH_MODE;
}

process_ellipse_move(x,y)
    int	x,y;
{
  if (move_mode==STRETCH_MODE)
    stretch_ellipse(selected_object,x,y,LOWER_RIGHT_HANDLE);
}

stretch_ellipse(ell,x,y,handle_no)
    Ellipse	*ell;
    int	x,y,handle_no;
{
  XPoint	delta;
  register int	sizex,sizey;

  diff_pos(&delta,&(ell->upper_left),x,y);
  draw_ellipse(ell);
  
  switch (handle_no) {
   case UPPER_LEFT_HANDLE:
    set_pos(&(ell->upper_left),x,y);
    dec_pos(&(ell->size),&delta);
    break;
   case UPPER_MIDDLE_HANDLE:
    ell->upper_left.y=y;
    ell->size.y-=delta.y;
    break;
   case UPPER_RIGHT_HANDLE:
    ell->upper_left.y=y;
    ell->size.x=delta.x;
    ell->size.y-=delta.y;
    break;
   case MIDDLE_RIGHT_HANDLE:
    ell->size.x=delta.x;
    break;
   case LOWER_RIGHT_HANDLE: 
    set_pos(&(ell->size),delta.x,delta.y);
    break;
   case LOWER_MIDDLE_HANDLE:
    ell->size.y=delta.y;
    break;
   case LOWER_LEFT_HANDLE:
    ell->upper_left.x=x;
    ell->size.x-=delta.x;
    ell->size.y=delta.y;
    break;
   case MIDDLE_LEFT_HANDLE:
    ell->upper_left.x=x;
    ell->size.x-=delta.x;
  }
  
  if (ell->size.x<0)
    ell->size.x=0;
  if (ell->size.y<0)
    ell->size.y=0;

  draw_ellipse(ell);
  set_pos(&move_org,x,y);
}


draw_ellipse(ell)
    Ellipse	*ell;
{
  int	ul_x;
  GC	gc;

  extern int get_object_color();

  if (reverse_mode)
    ul_x=reverse(ell->upper_left.x + ell->size.x);
  else
    ul_x=ell->upper_left.x;

  if (selected_object == (Object *) ell || is_analysis)
    if (!fill_mode)
      gc=image_xor_dash_gc;
    else
      gc=image_xor_stip_gc;
  else {
    if (!fill_mode) {
      gc=image_dash_gc;
      if (ell->header.strength!=(NUM_STRENGTHS-1)) {
	XSetLineAttributes(display, gc, 0, LineOnOffDash, CapButt, JoinMiter);
	XSetDashes(display, gc, 0, dashes[ell->header.strength], 2);
      }
      else
	XSetLineAttributes(display, gc, 0, LineSolid, CapButt, JoinMiter);
    }
    else {
      gc=image_stip_gc;
      XSetStipple(display, gc, image_stipple_pixmap[ell->header.strength]);
    }
    XSetForeground(display, gc, get_object_color(ell->header.type,
						 ell->header.inj_num));
  }

  if (!fill_mode)
    XDrawArc(display,image_win.window,gc,
	     ul_x, ell->upper_left.y,
	     ell->size.x,ell->size.y,0,FULL_ARC);
  else
    XFillArc(display,image_win.window,gc,
	     ul_x, ell->upper_left.y,
	     ell->size.x,ell->size.y,0,FULL_ARC);
}

draw_ellipse_in_icon(ell,image_num)
    Ellipse	*ell;
    int	image_num;
{
  XPoint	ul,size;

  extern int get_object_color();

  scale_pos(&ul,&(ell->upper_left),ICON_SF);
  scale_pos(&size,&(ell->size),ICON_SF);

  if (reverse_mode)
    ul.x=icon_reverse(ul.x + size.x, image_num);

  XSetForeground(display, icons_stip_gc,
		 get_object_color(ell->header.type, ell->header.inj_num));

  XSetStipple(display,icons_stip_gc,
	      icons_stipple_pixmap[ell->header.strength]);

  XFillArc(display, icons_win.window, icons_stip_gc,
	   icon_pos[image_num].x+ul.x, icon_pos[image_num].y+ul.y,
	   size.x, size.y, 0, FULL_ARC);
}

draw_ellipse_in_icon_pixmap(ell, fill)
    Ellipse	*ell;
{
  XPoint	ul,size;

  scale_pos(&ul,&(ell->upper_left),ICON_SF);
  scale_pos(&size,&(ell->size),ICON_SF);
  XSetForeground(display,pixmap_gc,ell->header.strength+1);
  if (fill)
    XFillArc(display, analyze_pixmap, pixmap_gc,
	     ul.x, ul.y, size.x, size.y, 0, FULL_ARC);
  else
    XDrawArc(display, analyze_pixmap, pixmap_gc,
	     ul.x, ul.y, size.x, size.y, 0, FULL_ARC);
}


/* 
 * handle bars are in clockwise order, starting with upper left corner
 */
set_ellipse_handle_bars(ell)
    Ellipse	*ell;
{
  XPoint	ul,size;
  int		i;

  set_pos(&ul,ell->upper_left.x-BAR_SIZE/2,ell->upper_left.y-BAR_SIZE/2);
  set_pos(&size,ell->size.x-1,ell->size.y-1);

  set_pos(&bar[0],ul.x,ul.y);
  set_pos(&bar[1],ul.x+size.x/2,ul.y);
  set_pos(&bar[2],ul.x+size.x,ul.y);
  set_pos(&bar[3],ul.x+size.x,ul.y+size.y/2);
  set_pos(&bar[4],ul.x+size.x,ul.y+size.y);
  set_pos(&bar[5],ul.x+size.x/2,ul.y+size.y);
  set_pos(&bar[6],ul.x,ul.y+size.y);
  set_pos(&bar[7],ul.x,ul.y+size.y/2);
  
  if (reverse_mode) {
    for (i=0; i<NUM_ELLIPSE_HANDLE_BARS; i++)
      bar[i].x = reverse(bar[i].x+BAR_SIZE);
  }
}
