/*
 * user interface event processing
 *
 * Copyright 1994, 1995 Bruno Olshausen and Bill Press
 * Washington University School of Medicine
 *
 */

#include <stdio.h>

#include <X11/keysym.h>

#include "xanat.h"
#include "ui.h"
#include "ui_text.h"
#include "ui_objects.h"

static int	done_p;



/*
 * main event loop
 */
main_event_loop()
{
  done_p=0;
  while (!done_p) {
    XNextEvent(display, &event);
    process_event(&event);
  }
}

process_event(event)
    XEvent	*event;
{
  switch ((int) event->type) {
   case ButtonPress:
    process_button_press(event);
    break;
   case ButtonRelease:
    process_button_release(event);
    break;
   case MotionNotify:
    process_button_motion(event);
    break;
   case LeaveNotify:
    process_leaving(event);
    break;
   case KeyPress:
    process_key_press(event);
    break;
   case Expose:
    process_expose(event);
    break;
  }
}


/*
 * button events
 */

process_button_press(button_event)
    XButtonEvent	*button_event;
{
  int		x,y,b;
  Window	win;

  win=button_event->window;
  b=button_event->button;
  x=button_event->x;
  y=button_event->y;

  wrote_message = 0;

  if (win==image_win.window)
    process_image_win_button(x,y,b);
  else if (win==control_win.window)
    process_control_win_button(x,y,b);
  else if (win==data_win.window)
    process_data_win_button(x,y);
  else if (win==icons_win.window)
    process_icons_win_button(x,y);
  else if (win==search_win.window)
    process_search_win_button(x,y);
  else if (win==index_win.window)
    process_index_win_button(x,y);

  if (wrote_message==0)
    redraw_message_win();
  else
    wrote_message=0;
}

process_control_win_button(x,y,b)
    int	x,y,b;
{
  int		index,x_index,y_index,sub_index;

  x_index=x/control_spacing.x;
  y_index=y/control_spacing.y;
  index=y_index*num_controls_per_row + x_index;
  if (index<NUM_CONTROLS) {
    if (control_list[is_stack][is_analysis][index].num_sub_choices)
      sub_index=submenu_index(x,x_index,index);
    switch(control_list[is_stack][is_analysis][index].id) {
     case QUIT_ID:
      done_p=quit_p();
      break;
     case LIST_ID:
      add_delete_list(sub_index);
      break;
     case ANALYSIS_METHOD_ID:
      set_analysis_method(sub_index);
      break;
     case VIEW_MODE_ID:
      set_view_mode(sub_index);
      break;
     case IMAGE_MODE_ID:
      set_image_mode(sub_index);
      break;
     case DRAW_MODE_ID:
      set_draw_mode(sub_index);
      break;
     case ANALYSIS_ID:
      do_analysis(sub_index);
      break;
     case FILE_ID:
      file_handling(sub_index);
      break;
     case STACK_ID:
      add_delete_stack(sub_index);
      break;
     case INJ_OR_LABEL_ID:
      set_inj_or_label_mode(sub_index);
      break;
     case INJ_TYPE_ID:
      switch (b) {
       case Button1:
	set_injection_type(sub_index);
	break;
       case Button3:
	toggle_injection(sub_index);
	break;
      }
      break;
     case WEIGHT_ID:
      set_label_weight(sub_index);
      break;
     case COMPARE_ID:
      compare_heatmaps(sub_index);
      break;
    }
    redraw_control_win();
  }
}

submenu_index(x,i,index)
    int	x,i,index;
{
  return((x-i*control_spacing.x)/
	 (control_spacing.x/
	  control_list[is_stack][is_analysis][index].num_sub_choices));
}


process_icons_win_button(x,y)
    int	x,y;
{
  int	i;

  for (i=0; i<num_images; i++) {
    if ((x>icon_pos[i].x && x<icon_pos[i].x+icon[i].width) &&
	(y>icon_pos[i].y && y<icon_pos[i].y+icon[i].height)) {
      select_image(i);
      break;
    }
  }
}

process_image_win_button(x,y,b)
    int	x,y,b;
{
  char	line[80];

  if (b==Button1 || (b==Button3 && !polygon_finished_p))
    if (selected_datum[view_mode])
      if (!selected_datum[view_mode]->writeProtect)
	if (selected_datum[view_mode]->validSlice[current_image])
	  drawing_press_action(x,y,b);
	else if (XPointInRegion(valid_region,x,y) &&
		 selected_datum[view_mode]->type==DATA_TYPE) {
	  toggle_valid_flag();
	  XMaskEvent(display,ButtonReleaseMask,&event);
	}
	else
	  perr("Must make slice valid to edit");
      else {
	sprintf(line,"Click \">%s\" to remove write-protection",
		selected_datum[view_mode]->ref.abbrev);
	perr(line);
      }
    else
      perr("Create record with List(add) or File(load)");
}

process_search_win_button(x,y)
    int	x,y;
{
  if (in_box_p(x,y,&search_box))
    data_search();
  else if (in_box_p(x,y,&analyze_box))
    do_list_analysis(&search_fields);
  else
    new_field(&search_win,x,y);
}


process_data_win_button(x,y)
    int	x,y;
{
  if (view_mode==DATA && selected_datum[DATA])
    new_field(&data_win,x,y);
}

process_index_win_button(x,y)
    int	x,y;
{
  if (in_box_p(x,y,&index_pageup_box))
    index_page_up();
  else if (in_box_p(x,y,&index_pagedown_box))
    index_page_down();
  else if (in_box_p(x,y,&index_stepup_box))
    datum_list_step_up();
  else if (in_box_p(x,y,&index_stepdown_box))
    datum_list_step_down();
  else if (y>INDEX_OFFSET_Y && x>INDEX_OFFSET_X)
    datum_select((y-INDEX_OFFSET_Y)/INDEX_ICON_HEIGHT);
}


/* button release */

process_button_release(release_event)
    XButtonReleasedEvent	*release_event;
{
  Window	win;

  win=release_event->window;

  if (win==image_win.window)
    process_image_win_release();
}

process_image_win_release()
{
  if (selected_datum[view_mode])
    if (selected_datum[view_mode]->validSlice[current_image] &&
	!selected_datum[view_mode]->writeProtect)
      drawing_release_action();
}



/*
 * motion
 */

process_button_motion(motion_event)
    XPointerMovedEvent	*motion_event;
{
  Window	win;
  int		x,y;

  win=motion_event->window;
  x=motion_event->x;
  y=motion_event->y;

  if (win==image_win.window)
    process_image_win_motion(x,y);
}

process_image_win_motion(x,y)
    int	x,y;
{
  if (selected_datum[view_mode])
    if (selected_datum[view_mode]->validSlice[current_image])
      drawing_move_action(x,y);

  if (validCoordP(current_image))
    update_coord(x,y);
}


/*
 * leaving
 */

process_leaving(leave_event)
    XLeaveWindowEvent	*leave_event;
{
  Window	win;

  win=leave_event->window;

  if (win==image_win.window)
    clear_coord();
}


/*
 * key events
 */
static char	kbuf[10];


process_key_press(key_event)
    XKeyEvent	*key_event;
{
  KeySym keysym;

  XLookupString(key_event, kbuf, sizeof(kbuf), &keysym, NULL);

  if (key_event->window==search_win.window && field_selected_p)
    process_text(&search_win,keysym,kbuf);
  else if (key_event->window==data_win.window && field_selected_p)
    process_text(&data_win,keysym,kbuf);
  else if (key_event->window==image_win.window)
    process_graphics_keypress(kbuf);
}

process_text(obj,keysym,ch)
    win_obj	*obj;
    KeySym	keysym;
    char	*ch;
{
  if (keysym==XK_Return)
    text_return(obj);
  else if ((keysym==XK_Delete) || (keysym==XK_BackSpace))
    backspace(obj);
  else if ((keysym>=XK_space)&&(keysym<=XK_asciitilde))
    process_character(obj,ch);
}

process_graphics_keypress(ch)
    char	*ch;
{
  switch (*ch) {
   case 'x': case 'X':
    copy_selected_object_to_buffer();
    delete_selected_object();
    break;
   case 'c': case 'C':
    copy_selected_object_to_buffer();
    break;
   case 'v': case 'V':
    if (selected_datum[view_mode])
      if (selected_datum[view_mode]->validSlice[current_image])
	paste_object_buffer_to_datum();
    break;
   case 'f': case 'F':
    pop_selected_object_to_front();
    break;
   case 'b': case 'B':
    push_selected_object_to_back();
    break;
  }
}


/*
 * Expose
 */
process_expose(expose_event)
    XExposeEvent	*expose_event;
{
  if (expose_event->window==control_win.window)
    redraw_control_win();
  else if (expose_event->window==icons_win.window)
    redraw_icons_win();
  else if (expose_event->window==image_win.window)
    redraw_image_win(0);
  else if (expose_event->window==search_win.window)
    redraw_search_win();
  else if (expose_event->window==data_win.window)
    redraw_data_win();
  else if (expose_event->window==index_win.window)
    redraw_index_win();
  else if (expose_event->window==message_win.window)
    redraw_message_win();
  else if (expose_event->window==coord_win.window)
    redraw_coord_win();
}
