/*
 * routines for manipulating objects
 *
 * Copyright 1994, 1995 Bill Press and Bruno Olshausen
 * Washington University School of Medicine
 *
 */

#include <stdio.h>
#include <malloc.h>

#include "xanat.h"
#include "ui.h"
#include "ui_objects.h"

XPoint	move_org,bar[MAX_NUM_POLYGON_VERTICES];
int	move_mode=NULL_MODE;
int	current_handle=NO_HANDLE;
int	polygon_finished_p=1;


static Object	object_buffer;
static int	object_buffer_valid_p=0;
static int	handle_bars_drawn=0;

/*
 * Generic object routines
 */

Object *which_object(x,y)
    int	x,y;
{
  Object	*obj;
  Object	*found_obj;

  obj=selected_datum[view_mode]->objects[current_image];
  found_obj=NULL;

  while (obj) {
    if (within_object_p(obj,x,y))
      found_obj=obj;
    obj=obj->header.next;
  }
  return(found_obj);
}

Object *object_exists(object_list, mode)
    Object	*object_list;
    int	mode;
{
  Object	*obj;

  obj=object_list;
  while(obj) {
    if (obj->header.type==mode)
      return(obj);
    obj=obj->header.next;
  }
  return(NULL);
}

object_size(obj)
    Object	*obj;
{
  int	i,j,area;
  Box	area_box;

  compute_object_icon_bounding_box(obj,&area_box);

  draw_object_in_icon_pixmap(obj, 1);
  get_data_ximage();

  area=0;

  for (j=area_box.ul.y; j<=area_box.lr.y; j++) {
    for (i=area_box.ul.x; i<=area_box.lr.x; i++) {
      if (data_ximage_pixel(i,j))
	area++;
    }
  }
  return(area);
}

within_object_p(obj,x,y)
    Object	*obj;
    int	x,y;
{
  if (obj->header.shape==ELLIPSE)
    return(within_ellipse_p(obj,x,y));
  else
    return(within_polygon_p(obj,x,y));
}

compare_objects(Object *obj1, Object *obj2)
{
  if (obj1->header.shape == obj2->header.shape)
    if (obj1->header.shape==ELLIPSE)
      return(compare_ellipses(&(obj1->ellipse),&(obj2->ellipse)));
    else
      return(compare_polygons(&(obj1->polygon),&(obj2->polygon)));
  else
    return (0);
}

copy_object(to_obj,from_obj)
    Object	*to_obj,*from_obj;
{
  int	i,n,shape;

  to_obj->header.type=from_obj->header.type;
  to_obj->header.strength = from_obj->header.strength;
  to_obj->header.inj_num = from_obj->header.inj_num;
  to_obj->header.size = from_obj->header.size;
  shape=to_obj->header.shape=from_obj->header.shape;

  if (shape==ELLIPSE) {
    copy_pos(&(to_obj->ellipse.upper_left),&(from_obj->ellipse.upper_left));
    copy_pos(&(to_obj->ellipse.size),&(from_obj->ellipse.size));
  }
  else {
    n=to_obj->polygon.num_points=from_obj->polygon.num_points;
    for (i=0; i<n; i++)
      copy_pos(&(to_obj->polygon.points[i]),&(from_obj->polygon.points[i]));
  }
}

int object_is_in_back(Object *obj)
{
  if (selected_datum[view_mode]->objects[current_image]==obj)
    return (1);
  else
    return (0);
}

int object_is_in_front(Object *obj)
{
  Object *o_ptr;

  if (!(o_ptr=selected_datum[view_mode]->objects[current_image]))
    return (0);
  else {
    while (o_ptr->header.next)
      o_ptr = o_ptr->header.next;
    if (o_ptr==obj)
      return (1);
    else
      return (0);
  }
}

/* Adding and deleting objects to and from datums */

add_object_to_top_datum_list(obj, datum)
    Object	*obj;
    Datum	*datum;
{
  Object *o_ptr;

  if (!(o_ptr=datum->objects[current_image]))
    datum->objects[current_image]=obj;
  else {
    while (o_ptr->header.next)
      o_ptr = o_ptr->header.next;
    o_ptr->header.next=obj;
    obj->header.next=NULL;
  }
}

add_object_to_bottom_datum_list(obj, datum)
    Object	*obj;
    Datum	*datum;
{
  obj->header.next=datum->objects[current_image];
  datum->objects[current_image]=obj;
}

remove_object_from_datum_list(remove_obj, datum)
    Object	*remove_obj;
    Datum	*datum;
{
  Object	*obj,*object_list;

  obj=object_list=datum->objects[current_image];

  if (obj==remove_obj)
    object_list=remove_obj->header.next;
  else {
    while (obj->header.next != remove_obj)
      obj=obj->header.next;
    obj->header.next=remove_obj->header.next;
  }
  free(remove_obj);

  datum->objects[current_image]=object_list;
}


/*
 * User-interface-related object routines
 */


/* selecting and deselecting objects */

deselect_and_erase_object()
{
  Object *obj;

  if (obj=selected_object) {
    if (object_is_in_front(obj)) {
      erase_handle_bars();
      draw_object(obj);
      selected_object=NULL;
      remove_object_from_datum_list(obj, selected_datum[view_mode]);
    }
    else {
      selected_object=NULL;
      remove_object_from_datum_list(obj, selected_datum[view_mode]);
      redraw_image_win(0);
    }
    redraw_icon(current_image);
    update_datum_areas();
  }
}

deselect_object()
{
  Object *obj;

  if (obj=selected_object)
    if (object_is_in_front(obj)) {
      erase_handle_bars();
      draw_object(obj);
      selected_object=NULL;
      draw_object(obj);
    }
    else {
      selected_object=NULL;
      redraw_image_win(0);
    }
}

select_object(obj)
    Object	*obj;
{
  object_mode=obj->header.type;
  inject_num=obj->header.inj_num;
  strength_num=obj->header.strength;
  set_xor_stipple();
  set_image_win_colors();
  redraw_control_win();
  selected_object=obj;
  set_handle_bars();
}

/* Drawing routines called by pressing mouse button */

drawing_press_action(x,y,b)
    int	x,y;
{
  if (reverse_mode)
    x=reverse(x);

  XGrabPointer(display, image_win.window, True, 0, GrabModeAsync,
	       GrabModeAsync, image_win.window, None, CurrentTime);

  switch (draw_mode) {
   case SELECTION_MODE:
    process_selection_press(x,y);
    break;
   case DRAW_ELLIPSE_MODE:
    process_ellipse_press(x,y);
    break;
   case DRAW_POLYGON_MODE:
    process_polygon_press(x,y,b);
    break;
   case DRAW_AREA_MODE:
    process_area_press(x,y);
    break;
  }
}

process_selection_press(x,y)
    int	x,y;
{
  Object	*obj;
  extern Object	*which_object();

  if (current_handle!=NO_HANDLE) {
    erase_handle_bars();
    move_mode=STRETCH_MODE;
  } 
  else if (obj=which_object(x,y)) {
    if (inject_toggles[obj->header.inj_num]) {
      if (obj!=selected_object) {
	erase_handle_bars();
	select_object(obj);
	if (object_is_in_front(selected_object))
	  redraw_image_win(0);
      }
      pop_selected_object_to_front();
      erase_handle_bars();
      set_pos(&move_org,x,y);
      move_mode=MOVE_OBJECT_MODE;
    }
  }
  else if ((!reverse_mode && XPointInRegion(valid_region,x,y)) ||
	   (reverse_mode && XPointInRegion(valid_region,reverse(x),y))) {
    if (selected_datum[view_mode]->type==DATA_TYPE) {
      if (reverse_mode)
	x=reverse(x);
      if (XPointInRegion(valid_region,x,y)) {
	toggle_valid_flag();
	XUngrabPointer(display, CurrentTime);
      }
    }
  }
  else
    deselect_object();
}

new_object(x,y)
    int	x,y;
{
  Object	*new_obj;

  new_obj=(Object *)calloc(1,sizeof(Object));
  MCHECK(new_obj);

  if (draw_mode==DRAW_ELLIPSE_MODE) {
    new_obj->header.shape=ELLIPSE;
    init_ellipse(new_obj,x,y);
  }
  else {
    new_obj->header.shape=POLYGON;
    init_polygon(new_obj,x,y);
  }

  if (selected_datum[view_mode]->type==ANALYSIS_TYPE)
    object_mode=SEARCH_AREA;

  new_obj->header.type=object_mode;
  new_obj->header.strength=strength_num;
  new_obj->header.inj_num=inject_num;
  
  add_object_to_top_datum_list(new_obj,selected_datum[view_mode]);

  draw_object(new_obj);
  deselect_object();
  select_object(new_obj);
}


/* Routines called while mouse button being held down */

drawing_move_action(x,y)
    int	x,y;
{
  if (reverse_mode)
    x=reverse(x);

  switch (draw_mode) {
   case SELECTION_MODE:
    process_selection_move(x,y);
    break;
   case DRAW_ELLIPSE_MODE:
    process_ellipse_move(x,y);
    break;
   case DRAW_POLYGON_MODE:
    process_polygon_move(x,y);
    break;
  }
}

process_selection_move(x,y)
    int	x,y;
{
  switch (move_mode) {
   case NULL_MODE:
    if (selected_object)
      check_within_handles((reverse_mode) ? reverse(x) : x, y);
    break;
   case MOVE_OBJECT_MODE:
    move_object(x,y);
    break;
   case STRETCH_MODE:
    stretch_object(x,y);
    break;
  }
}

move_object(x,y)
    int	x,y;
{
  XPoint	delta;

  diff_pos(&delta,&move_org,x,y);
  draw_object(selected_object);
  if (selected_object->header.shape==ELLIPSE)
    inc_ellipse_pos(selected_object,&delta);
  else
    inc_polygon_pos(selected_object,&delta);
  draw_object(selected_object);
  set_pos(&move_org,x,y);
}

check_within_handles(x,y)
    int	x,y;
{
  static int	currently_in_handles_p=0;

  if ((current_handle=which_handle(x,y))!=NO_HANDLE) {
    if (!currently_in_handles_p) {
      XDefineCursor(display,image_win.window,cross_cursor);
      currently_in_handles_p=1;
    }
  }
  else {
    if (currently_in_handles_p) {
      XDefineCursor(display,image_win.window,arrow_cursor);
      currently_in_handles_p=0;
    }
  }
}

which_handle(x,y)
    int	x,y;
{
  register int		i,num_handle_bars;
  static XPoint	bar_size={BAR_SIZE,BAR_SIZE};

  num_handle_bars = (selected_object->header.shape==ELLIPSE) ?
    NUM_ELLIPSE_HANDLE_BARS : num_polygon_handles(selected_object);

  for (i=0; i<num_handle_bars; i++) {
    if (in_region_p(&bar[i],&bar_size,x,y))
      return(i);
  }
  return(NO_HANDLE);
}

stretch_object(x,y)
    int	x,y;
{
  if (selected_object->header.shape==ELLIPSE)
    stretch_ellipse(selected_object,x,y,current_handle);
  else
    stretch_polygon(selected_object,x,y,current_handle);
}    

/* Routines called when mouse button is released */

drawing_release_action()
{
  switch (draw_mode) {
   case SELECTION_MODE:
    if (selected_object)
      let_go_selected_object();
    break;
   case DRAW_ELLIPSE_MODE:
    let_go_selected_object();
    break;
   case DRAW_POLYGON_MODE:
    if (polygon_finished_p)
      let_go_selected_object();
    break;
  }

  if (polygon_finished_p)
    XUngrabPointer(display, CurrentTime);
}

let_go_selected_object()
{
  update_datum_areas();
  set_handle_bars();
  draw_handle_bars();
  redraw_icon(current_image);
  if (draw_mode != SELECTION_MODE || move_mode != NULL_MODE) {
    selected_object->header.size=object_size(selected_object);
    draw_mode=SELECTION_MODE;
    move_mode=NULL_MODE;
    redraw_control_win();
  }
}

draw_object(obj)
    Object	*obj;
{
  if (inject_toggles[obj->header.inj_num] || is_analysis)
    if (obj->header.shape==ELLIPSE)
      draw_ellipse(obj);
    else
      draw_polygon(obj);
}

draw_object_in_icon(obj,image_num)
    Object	*obj;
    int	image_num;
{
  if (inject_toggles[obj->header.inj_num] || is_analysis)
    if (obj->header.shape==ELLIPSE)
      draw_ellipse_in_icon(obj,image_num);
    else
      draw_polygon_in_icon(obj,image_num);
}

set_handle_bars()
{
  if (selected_object->header.shape==ELLIPSE)
    set_ellipse_handle_bars(selected_object);
  else
    set_polygon_handle_bars(selected_object);
}

clear_handle_var()
{
  handle_bars_drawn=0;
}

draw_handle_bars()
{
  int	i,num_handle_bars;

  if (!handle_bars_drawn) {

    num_handle_bars = (selected_object->header.shape==ELLIPSE) ?
      NUM_ELLIPSE_HANDLE_BARS : num_polygon_handles(selected_object);

    for (i=0; i<num_handle_bars; i++)
      XFillRectangle(display,image_win.window,image_handle_gc,
		     bar[i].x,bar[i].y,BAR_SIZE,BAR_SIZE);

    handle_bars_drawn=1;
  }
}


erase_handle_bars()
{
  int	i,num_handle_bars;

  if (handle_bars_drawn) {

    num_handle_bars = (selected_object->header.shape==ELLIPSE) ?
      NUM_ELLIPSE_HANDLE_BARS : num_polygon_handles(selected_object);

    for (i=0; i<num_handle_bars; i++)
      XFillRectangle(display,image_win.window,image_handle_gc,
		     bar[i].x,bar[i].y,BAR_SIZE,BAR_SIZE);

    handle_bars_drawn=0;
  }
}


/* editing objects in image win with keystrokes */

delete_selected_object()
{
  if (selected_object) {
    deselect_and_erase_object();
    if (current_handle!=NO_HANDLE) {
      XDefineCursor(display,image_win.window,arrow_cursor);
      current_handle=NO_HANDLE;
    }
  }
  else
    perr("No objected selected to delete");
}

copy_selected_object_to_buffer()
{
  if (selected_object) {
    copy_object(&object_buffer,selected_object);
    object_buffer_valid_p=1;
    pmes("Object copied to buffer");
  }
  else
    perr("No object selected to copy");
}

paste_object_buffer_to_datum()
{
  Object	*new_obj;

  if (object_buffer_valid_p) {
    new_obj=(Object *)calloc(1,sizeof(Object));
    MCHECK(new_obj);
    copy_object(new_obj,&object_buffer);
    if (selected_datum[view_mode]->type==ANALYSIS_TYPE)
      new_obj->header.type=SEARCH_AREA;
    add_object_to_top_datum_list(new_obj,selected_datum[view_mode]);
    update_datum_areas();
    deselect_object();
    select_object(new_obj);
    redraw_icon(current_image);
    reset_image_win();
  }
  else {
    perr("Nothing in copy buffer");
  }
}

pop_selected_object_to_front()
{
  Object	*new_obj;

  if (selected_object) {
    if (!object_is_in_front(selected_object)) {
      new_obj=(Object *)calloc(1,sizeof(Object));
      MCHECK(new_obj);
      copy_object(new_obj,selected_object);
      remove_object_from_datum_list(selected_object,selected_datum[view_mode]);
      add_object_to_top_datum_list(new_obj,selected_datum[view_mode]);
      select_object(new_obj);
      redraw_icon(current_image);
      reset_image_win();
    }
  }
}

push_selected_object_to_back()
{
  Object	*new_obj;

  if (selected_object)
    if (!object_is_in_back(selected_object)) {
      new_obj=(Object *)calloc(1,sizeof(Object));
      MCHECK(new_obj);
      copy_object(new_obj,selected_object);
      remove_object_from_datum_list(selected_object,selected_datum[view_mode]);
      add_object_to_bottom_datum_list(new_obj,selected_datum[view_mode]);
      select_object(new_obj);
      redraw_icon(current_image);
      reset_image_win();
    }
}
